<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_role(['ADMIN','STOREKEEPER']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

$hasVendors = table_exists($pdo, 'vendors') && column_exists($pdo, 'purchases', 'vendor_id');
if (!$hasVendors) {
  flash_set('danger', 'Vendors table / purchases.vendor_id not found. Run the SQL upgrade first.');
  header('Location: ' . base_url('admin/purchases_new.php'));
  exit;
}

$hasInvoiceDate = column_exists($pdo, 'purchases', 'invoice_date');
$hasInvoicesTable = table_exists($pdo, 'purchase_invoices');
$hasVatMeta = column_exists($pdo, 'purchases', 'vat_included') && column_exists($pdo, 'purchases', 'vat_percent');
$hasTotals = column_exists($pdo, 'purchases', 'subtotal') && column_exists($pdo, 'purchases', 'vat_total') && column_exists($pdo, 'purchases', 'grand_total');
$hasVatRateId = column_exists($pdo, 'purchases', 'vat_rate_id');
$hasLineVat = column_exists($pdo, 'purchase_lines', 'vat_percent') && column_exists($pdo, 'purchase_lines', 'vat_amount') && column_exists($pdo, 'purchase_lines', 'line_total');

// VAT rate (current)
$vatRateId = null;
$vatPercentGlobal = 0.0;
if (table_exists($pdo, 'vat_rates')) {
  $row = $pdo->query("SELECT id, vat_percent FROM vat_rates WHERE effective_from <= CURDATE() ORDER BY effective_from DESC LIMIT 1")->fetch();
  if ($row) { $vatRateId = (int)$row['id']; $vatPercentGlobal = (float)$row['vat_percent']; }
}
$vatIncluded = isset($_POST['vat_included']) ? 1 : 0;

// Header fields
$invoiceNo = trim($_POST['invoice_no'] ?? '');
$invoiceDate = trim($_POST['invoice_date'] ?? '');
$note = trim($_POST['note'] ?? '');

// Vendor
$vendorId = (int)($_POST['vendor_id'] ?? 0);
if ($vendorId <= 0) {
  flash_set('danger', 'Please select a vendor.');
  header('Location: ' . base_url('admin/purchases_new.php'));
  exit;
}

// Lines
$itemIds = $_POST['item_id'] ?? [];
$qtys = $_POST['qty'] ?? [];
$unitCosts = $_POST['unit_cost'] ?? [];
$applyVatArr = $_POST['apply_vat'] ?? [];

if (!is_array($itemIds) || count($itemIds) === 0) {
  flash_set('danger', 'Please add at least one purchase line.');
  header('Location: ' . base_url('admin/purchases_new.php'));
  exit;
}

// Upload multiple invoice files
$uploadedFiles = [];
if (isset($_FILES['invoice_files']) && is_array($_FILES['invoice_files']['name'])) {
  $maxBytes = 8 * 1024 * 1024; // 8MB each
  $allowed = ['pdf','jpg','jpeg','png'];

  $dir = __DIR__ . '/../../uploads/invoices';
  if (!is_dir($dir)) @mkdir($dir, 0775, true);

  for ($k=0; $k<count($_FILES['invoice_files']['name']); $k++) {
    $name = $_FILES['invoice_files']['name'][$k];
    $tmp = $_FILES['invoice_files']['tmp_name'][$k];
    $err = $_FILES['invoice_files']['error'][$k];
    $size = $_FILES['invoice_files']['size'][$k];

    if ($err === UPLOAD_ERR_NO_FILE || $name === '') continue;
    if ($err !== UPLOAD_ERR_OK) {
      flash_set('danger', 'One of the invoice files failed to upload.');
      header('Location: ' . base_url('admin/purchases_new.php'));
      exit;
    }
    if ($size > $maxBytes) {
      flash_set('danger', 'One invoice file is too large (max 8MB each).');
      header('Location: ' . base_url('admin/purchases_new.php'));
      exit;
    }
    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    if (!in_array($ext, $allowed, true)) {
      flash_set('danger', 'Invalid invoice file type. Use PDF/JPG/PNG.');
      header('Location: ' . base_url('admin/purchases_new.php'));
      exit;
    }

    $safeName = 'inv_' . date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
    $dest = $dir . '/' . $safeName;
    if (!move_uploaded_file($tmp, $dest)) {
      flash_set('danger', 'Could not save one of the invoice files.');
      header('Location: ' . base_url('admin/purchases_new.php'));
      exit;
    }

    $uploadedFiles[] = [
      'file_path' => 'uploads/invoices/' . $safeName,
      'original_name' => $name,
    ];
  }
}

// Compute totals and prepare inserts
$lines = [];
$subtotal = 0.0;
$vatTotal = 0.0;
$grandTotal = 0.0;

for ($i=0; $i<count($itemIds); $i++) {
  $itemId = (int)$itemIds[$i];
  if ($itemId <= 0) continue;

  $qty = (int)($qtys[$i] ?? 0);
  if ($qty <= 0) $qty = 1;

  $priceInput = (float)($unitCosts[$i] ?? 0);
  if ($priceInput < 0) $priceInput = 0;

  $applyVat = isset($applyVatArr[$i]) && (string)$applyVatArr[$i] === '1';
  $vatPercent = $applyVat ? $vatPercentGlobal : 0.0;

  $unitNet = $priceInput;
  if ($applyVat && $vatIncluded && $vatPercent > 0) {
    $unitNet = $priceInput / (1 + ($vatPercent/100.0));
  }

  $lineSub = $qty * $unitNet;
  $lineVat = $lineSub * ($vatPercent/100.0);
  $lineTotal = $lineSub + $lineVat;

  $subtotal += $lineSub;
  $vatTotal += $lineVat;
  $grandTotal += $lineTotal;

  $lines[] = [
    'item_id' => $itemId,
    'qty' => $qty,
    'unit_cost' => round($unitNet, 2),
    'vat_percent' => $vatPercent,
    'vat_amount' => round($lineVat, 2),
    'line_total' => round($lineTotal, 2),
  ];
}

if (count($lines) === 0) {
  flash_set('danger', 'Please select items for the purchase.');
  header('Location: ' . base_url('admin/purchases_new.php'));
  exit;
}

$pdo->beginTransaction();
try {
  // Insert purchase header (match your existing schema)
  $cols = ['invoice_no','note','vendor_id'];
  $vals = [$invoiceNo, ($note===''?null:$note), $vendorId];

  // keep vendor_name column null (vendor comes from table)
  if (column_exists($pdo,'purchases','vendor_name')) {
    $cols[] = 'vendor_name';
    $vals[] = null;
  }

  // purchased_by / created_by
  if (column_exists($pdo,'purchases','purchased_by')) {
    $cols[] = 'purchased_by';
    $vals[] = current_user()['id'];
  } elseif (column_exists($pdo,'purchases','created_by')) {
    $cols[] = 'created_by';
    $vals[] = current_user()['id'];
  }

  if ($hasInvoiceDate) {
    $cols[] = 'invoice_date';
    $vals[] = ($invoiceDate===''?null:$invoiceDate);
  }

  if ($hasVatMeta) {
    $cols[] = 'vat_included';
    $vals[] = $vatIncluded;
    $cols[] = 'vat_percent';
    $vals[] = $vatPercentGlobal;
  }
  if ($hasVatRateId) {
    $cols[] = 'vat_rate_id';
    $vals[] = $vatRateId;
  }
  if ($hasTotals) {
    $cols[] = 'subtotal';    $vals[] = round($subtotal,2);
    $cols[] = 'vat_total';   $vals[] = round($vatTotal,2);
    $cols[] = 'grand_total'; $vals[] = round($grandTotal,2);
  }

  $placeholders = implode(',', array_fill(0, count($cols), '?'));
  $sql = "INSERT INTO purchases(" . implode(',', $cols) . ") VALUES($placeholders)";
  $pdo->prepare($sql)->execute($vals);
  $purchaseId = (int)$pdo->lastInsertId();

  // Insert lines + update stock
  if ($hasLineVat) {
    $insLine = $pdo->prepare("INSERT INTO purchase_lines(purchase_id,item_id,qty,unit_cost,vat_percent,vat_amount,line_total) VALUES(?,?,?,?,?,?,?)");
  } else {
    $insLine = $pdo->prepare("INSERT INTO purchase_lines(purchase_id,item_id,qty,unit_cost) VALUES(?,?,?,?)");
  }

  $updItem = $pdo->prepare("UPDATE items SET stock_qty = stock_qty + ? WHERE id=?");

  $insMove = null;
  if (table_exists($pdo,'stock_moves')) {
    $insMove = $pdo->prepare("INSERT INTO stock_moves(item_id, move_type, ref_table, ref_id, qty, created_by, note) VALUES(?, 'PURCHASE_IN','purchases',?,?,?,?)");
  }

  foreach ($lines as $ln) {
    if ($hasLineVat) {
      $insLine->execute([$purchaseId, $ln['item_id'], $ln['qty'], $ln['unit_cost'], $ln['vat_percent'], $ln['vat_amount'], $ln['line_total']]);
    } else {
      $insLine->execute([$purchaseId, $ln['item_id'], $ln['qty'], $ln['unit_cost']]);
    }
    $updItem->execute([$ln['qty'], $ln['item_id']]);

    if ($insMove) {
      $insMove->execute([$ln['item_id'], $purchaseId, $ln['qty'], current_user()['id'], "Purchase #{$purchaseId}"]);
    }
  }

  // Save invoice attachments
  if ($hasInvoicesTable && count($uploadedFiles) > 0) {
    $insInv = $pdo->prepare("INSERT INTO purchase_invoices(purchase_id,file_path,original_name,uploaded_by) VALUES(?,?,?,?)");
    foreach ($uploadedFiles as $f) {
      $insInv->execute([$purchaseId, $f['file_path'], $f['original_name'], current_user()['id']]);
    }
  } elseif (column_exists($pdo,'purchases','invoice_file') && count($uploadedFiles) > 0) {
    // fallback: store first file into purchases.invoice_file if exists
    $pdo->prepare("UPDATE purchases SET invoice_file=? WHERE id=?")->execute([$uploadedFiles[0]['file_path'], $purchaseId]);
  }

  $pdo->commit();
  flash_set('success', "Purchase #{$purchaseId} saved. Stock added.");
  header('Location: ' . base_url('admin/purchases.php'));
  exit;

} catch (Throwable $e) {
  $pdo->rollBack();
  flash_set('danger', 'Error saving purchase: ' . $e->getMessage());
  header('Location: ' . base_url('admin/purchases_new.php'));
  exit;
}
