<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['ADMIN','STOREKEEPER']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) {
  flash_set('danger', 'Invalid purchase id.');
  header('Location: ' . base_url('admin/purchases.php'));
  exit;
}

$hasVendors = table_exists($pdo,'vendors') && column_exists($pdo,'purchases','vendor_id');
$hasInvoiceDate = column_exists($pdo,'purchases','invoice_date');
$hasInvoicesTable = table_exists($pdo,'purchase_invoices');
$hasVatMeta = column_exists($pdo,'purchases','vat_included') && column_exists($pdo,'purchases','vat_percent');
$hasTotals = column_exists($pdo,'purchases','subtotal') && column_exists($pdo,'purchases','vat_total') && column_exists($pdo,'purchases','grand_total');

$hasCategories = table_exists($pdo,'categories') && column_exists($pdo,'items','category_id');
$hasLineVat = column_exists($pdo,'purchase_lines','vat_percent') && column_exists($pdo,'purchase_lines','vat_amount') && column_exists($pdo,'purchase_lines','line_total');

// Header
$sql = "
  SELECT
    p.*,
    COALESCE(u.full_name,'') AS created_by_name
    " . ($hasVendors ? ", v.name AS vendor_display" : "") . "
  FROM purchases p
  LEFT JOIN users u ON u.id = COALESCE(p.purchased_by, p.created_by)
  " . ($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "") . "
  WHERE p.id = ?
  LIMIT 1
";
$st = $pdo->prepare($sql);
$st->execute([$id]);
$p = $st->fetch();
if (!$p) {
  flash_set('danger', 'Purchase not found.');
  header('Location: ' . base_url('admin/purchases.php'));
  exit;
}

$vendor = $hasVendors ? ($p['vendor_display'] ?? '—') : ($p['vendor_name'] ?? '—');
$invoiceNo = $p['invoice_no'] ?? '';
$invoiceDate = $hasInvoiceDate ? ($p['invoice_date'] ?? null) : null;

$vatIncluded = $hasVatMeta ? ((int)$p['vat_included'] === 1) : false;
$vatPercent = $hasVatMeta ? (float)($p['vat_percent'] ?? 0) : 0.0;

// Lines
$linesSql = "
  SELECT
    pl.*,
    i.name AS item_name,
    i.unit AS item_unit
    " . ($hasCategories ? ", c.name AS category_name" : "") . "
  FROM purchase_lines pl
  JOIN items i ON i.id = pl.item_id
  " . ($hasCategories ? "LEFT JOIN categories c ON c.id = i.category_id" : "") . "
  WHERE pl.purchase_id = ?
  ORDER BY " . ($hasCategories ? "c.name, " : "") . " i.name
";
$lst = $pdo->prepare($linesSql);
$lst->execute([$id]);
$lines = $lst->fetchAll();

// Attachments
$files = [];
if ($hasInvoicesTable) {
  $fst = $pdo->prepare("SELECT file_path, original_name, uploaded_at FROM purchase_invoices WHERE purchase_id=? ORDER BY id DESC");
  $fst->execute([$id]);
  $files = $fst->fetchAll();
} elseif (column_exists($pdo,'purchases','invoice_file') && !empty($p['invoice_file'])) {
  $files = [[
    'file_path' => $p['invoice_file'],
    'original_name' => basename($p['invoice_file']),
    'uploaded_at' => $p['created_at'] ?? null,
  ]];
}

// Totals
$subtotal = 0.0; $vatTotal = 0.0; $grandTotal = 0.0;
foreach ($lines as $ln) {
  $qty = (int)$ln['qty'];
  $unit = (float)($ln['unit_cost'] ?? 0);
  $lineSub = $qty * $unit;
  $lineVat = 0.0;
  $lineTotal = $lineSub;
  if ($hasLineVat) {
    $lineVat = (float)($ln['vat_amount'] ?? 0);
    $lineTotal = (float)($ln['line_total'] ?? ($lineSub + $lineVat));
  } else {
    // fallback: apply header VAT percent if present
    $lineVat = $lineSub * ($vatPercent/100.0);
    $lineTotal = $lineSub + $lineVat;
  }
  $subtotal += $lineSub;
  $vatTotal += $lineVat;
  $grandTotal += $lineTotal;
}
if ($hasTotals) {
  $subtotal = (float)($p['subtotal'] ?? $subtotal);
  $vatTotal = (float)($p['vat_total'] ?? $vatTotal);
  $grandTotal = (float)($p['grand_total'] ?? $grandTotal);
}

include __DIR__ . '/../partials/header.php';
?>

<style>
@media print {
  nav.navbar, .no-print { display:none !important; }
  main.container { max-width: 100% !important; padding: 0 !important; }
  .card { box-shadow: none !important; }
}
</style>

<div class="d-flex justify-content-between align-items-start mb-3 no-print">
  <div>
    <h3 class="mb-1">Purchase #<?=h($id)?></h3>
    <div class="text-muted">Details • Currency: <strong>USD ($)</strong></div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-secondary" href="<?=h(base_url('admin/purchases.php'))?>">← Back</a>
    <button class="btn btn-primary" onclick="window.print()">Print</button>
  </div>
</div>

<div class="row g-3">
  <div class="col-lg-4">
    <div class="card shadow-sm">
      <div class="card-body">
        <h5 class="mb-3">Header</h5>
        <div class="mb-2"><span class="text-muted">Vendor:</span> <div class="fw-semibold"><?=h($vendor ?: '—')?></div></div>
        <div class="mb-2"><span class="text-muted">Invoice No:</span> <div class="fw-semibold"><?=h($invoiceNo ?: '—')?></div></div>
        <?php if ($hasInvoiceDate): ?>
          <div class="mb-2"><span class="text-muted">Invoice Date:</span> <div class="fw-semibold"><?=h($invoiceDate ?: '—')?></div></div>
        <?php endif; ?>
        <div class="mb-2"><span class="text-muted">Purchased at:</span> <div class="fw-semibold"><?=h($p['purchased_at'] ?? $p['created_at'] ?? '—')?></div></div>
        <div class="mb-2"><span class="text-muted">By:</span> <div class="fw-semibold"><?=h($p['created_by_name'] ?: '—')?></div></div>

        <div class="mt-3">
          <?php if ($vatPercent > 0): ?>
            <span class="badge bg-secondary"><?=h(number_format($vatPercent,2))?>% VAT</span>
          <?php endif; ?>
          <?php if ($vatIncluded): ?>
            <span class="badge bg-info text-dark">Prices incl. VAT</span>
          <?php else: ?>
            <span class="badge bg-light text-dark border">Prices excl. VAT</span>
          <?php endif; ?>
        </div>

        <?php if (!empty($p['note'])): ?>
          <hr>
          <div class="text-muted small">Note</div>
          <div><?=nl2br(h($p['note']))?></div>
        <?php endif; ?>
      </div>
    </div>

    <div class="card shadow-sm mt-3">
      <div class="card-body">
        <h5 class="mb-3">Invoice files</h5>
        <?php if (!$files): ?>
          <div class="text-muted">No files attached.</div>
        <?php else: ?>
          <div class="list-group">
            <?php foreach ($files as $f): ?>
              <a class="list-group-item list-group-item-action d-flex justify-content-between align-items-center"
                 href="<?=h(base_url($f['file_path']))?>" target="_blank">
                <div>
                  <div class="fw-semibold"><?=h($f['original_name'] ?: basename($f['file_path']))?></div>
                  <div class="text-muted small"><?=h($f['uploaded_at'] ?? '')?></div>
                </div>
                <span class="badge bg-secondary">Open</span>
              </a>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <div class="col-lg-8">
    <div class="card shadow-sm">
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table table-hover align-middle mb-0">
            <thead class="table-light">
              <tr>
                <?php if ($hasCategories): ?><th style="width:200px;">Category</th><?php endif; ?>
                <th>Item</th>
                <th style="width:90px;">Qty</th>
                <th style="width:130px;">Unit</th>
                <th style="width:150px;">Unit price</th>
                <th style="width:140px;">VAT</th>
                <th style="width:150px;">Line total</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$lines): ?>
                <tr><td colspan="<?=h($hasCategories?7:6)?>" class="p-4 text-center text-muted">No lines.</td></tr>
              <?php endif; ?>

              <?php foreach ($lines as $ln): ?>
                <?php
                  $qty = (int)$ln['qty'];
                  $unit = (float)($ln['unit_cost'] ?? 0);
                  $lineSub = $qty * $unit;
                  $lineVat = $hasLineVat ? (float)($ln['vat_amount'] ?? 0) : ($lineSub * ($vatPercent/100.0));
                  $lineTotal = $hasLineVat ? (float)($ln['line_total'] ?? ($lineSub+$lineVat)) : ($lineSub + $lineVat);
                ?>
                <tr>
                  <?php if ($hasCategories): ?><td class="text-muted"><?=h($ln['category_name'] ?? '—')?></td><?php endif; ?>
                  <td class="fw-semibold"><?=h($ln['item_name'])?> <span class="text-muted"> (<?=h($ln['item_unit'])?>)</span></td>
                  <td class="fw-semibold"><?=h($qty)?></td>
                  <td class="text-muted"><?=h($ln['item_unit'])?></td>
                  <td class="text-muted">$<?=h(number_format($unit,2))?></td>
                  <td class="text-muted">$<?=h(number_format($lineVat,2))?></td>
                  <td class="fw-semibold">$<?=h(number_format($lineTotal,2))?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
            <tfoot class="table-light">
              <tr>
                <th colspan="<?=h($hasCategories?5:4)?>" class="text-end">Subtotal</th>
                <th colspan="2" class="text-end">$<?=h(number_format($subtotal,2))?></th>
              </tr>
              <tr>
                <th colspan="<?=h($hasCategories?5:4)?>" class="text-end">VAT Total</th>
                <th colspan="2" class="text-end">$<?=h(number_format($vatTotal,2))?></th>
              </tr>
              <tr>
                <th colspan="<?=h($hasCategories?5:4)?>" class="text-end">Grand Total</th>
                <th colspan="2" class="text-end fw-semibold">$<?=h(number_format($grandTotal,2))?></th>
              </tr>
            </tfoot>
          </table>
        </div>
      </div>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../partials/footer.php'; ?>
