<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['ADMIN','STOREKEEPER']);

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}
function build_query_string(array $params): string {
  $merged = array_merge($_GET, $params);
  foreach ($merged as $k => $v) {
    if ($v === '' || $v === null) unset($merged[$k]);
  }
  return http_build_query($merged);
}

$hasVendors = table_exists($pdo,'vendors') && column_exists($pdo,'purchases','vendor_id');
$hasInvoiceDate = column_exists($pdo,'purchases','invoice_date');
$hasInvoicesTable = table_exists($pdo,'purchase_invoices');
$hasVatMeta = column_exists($pdo,'purchases','vat_included') && column_exists($pdo,'purchases','vat_percent');
$hasTotals = column_exists($pdo,'purchases','subtotal') && column_exists($pdo,'purchases','vat_total') && column_exists($pdo,'purchases','grand_total');
$hasCategories = table_exists($pdo,'categories') && column_exists($pdo,'items','category_id');

$vatPercentCurrent = 0.0;
if (table_exists($pdo, 'vat_rates')) {
  $vatPercentCurrent = (float)($pdo->query("SELECT vat_percent FROM vat_rates WHERE effective_from <= CURDATE() ORDER BY effective_from DESC LIMIT 1")->fetchColumn() ?: 0);
}

$q = trim($_GET['q'] ?? '');
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = (int)($_GET['per_page'] ?? 10);
if (!in_array($perPage, [5,10,20,50], true)) $perPage = 10;

$where = "1=1";
$params = [];
if ($q !== '') {
  $where .= " AND (
    CAST(p.id AS CHAR) LIKE ?
    OR COALESCE(p.invoice_no, '') LIKE ?
    OR COALESCE(" . ($hasVendors ? "v.name" : "p.vendor_name") . ", '') LIKE ?
    OR COALESCE(u.full_name, '') LIKE ?
  )";
  $like = "%{$q}%";
  $params = [$like, $like, $like, $like];
}

$countSql = "
  SELECT COUNT(*) AS total
  FROM purchases p
  LEFT JOIN users u ON u.id = COALESCE(p.purchased_by, p.created_by)
  " . ($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "") . "
  WHERE $where
";
$countStmt = $pdo->prepare($countSql);
$countStmt->execute($params);
$total = (int)($countStmt->fetchColumn() ?: 0);
$totalPages = max(1, (int)ceil($total / $perPage));
if ($page > $totalPages) $page = $totalPages;
$offset = ($page - 1) * $perPage;

$itemPrefix = $hasCategories ? "CONCAT(COALESCE(c.name,'—'), ': ', i.name)" : "i.name";
$itemsConcat = "GROUP_CONCAT(CONCAT($itemPrefix, ' ×', pl.qty, ' @ $', FORMAT(COALESCE(pl.unit_cost,0),2)) ORDER BY " . ($hasCategories ? "c.name, " : "") . " i.name SEPARATOR ', ')";

$subtotalExpr = $hasTotals ? "MAX(p.subtotal)" : "COALESCE(SUM(pl.qty * COALESCE(pl.unit_cost,0)),0)";
$vatExpr = $hasTotals ? "MAX(p.vat_total)" : "COALESCE(SUM(COALESCE(pl.vat_amount,0)),0)";
$totalExpr = $hasTotals ? "MAX(p.grand_total)" : "COALESCE(SUM(COALESCE(pl.line_total, (pl.qty * COALESCE(pl.unit_cost,0)) + COALESCE(pl.vat_amount,0))),0)";
$invConcat = $hasInvoicesTable ? "GROUP_CONCAT(CONCAT(pi.file_path,'::',COALESCE(pi.original_name,'')) SEPARATOR '||')" : "''";

$sql = "
  SELECT
    p.id,
    MAX(COALESCE(p.purchased_at, p.created_at)) AS purchased_at,
    MAX(COALESCE(p.purchase_date, DATE(COALESCE(p.purchased_at, p.created_at)))) AS purchase_date,
    MAX(p.invoice_no) AS invoice_no,
    " . ($hasInvoiceDate ? "MAX(p.invoice_date) AS invoice_date," : "NULL AS invoice_date,") . "
    MAX(" . ($hasVendors ? "v.name" : "p.vendor_name") . ") AS vendor_display,
    MAX(COALESCE(u.full_name, '')) AS created_by_name,
    MAX(" . ($hasVatMeta ? "p.vat_included" : "0") . ") AS vat_included,
    MAX(" . ($hasVatMeta ? "p.vat_percent" : $vatPercentCurrent) . ") AS vat_percent,
    COALESCE(SUM(pl.qty),0) AS total_qty,
    $subtotalExpr AS subtotal,
    $vatExpr AS vat_total,
    $totalExpr AS grand_total,
    COALESCE($itemsConcat, '') AS items_summary,
    COALESCE($invConcat, '') AS invoice_files
  FROM purchases p
  LEFT JOIN users u ON u.id = COALESCE(p.purchased_by, p.created_by)
  " . ($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "") . "
  LEFT JOIN purchase_lines pl ON pl.purchase_id = p.id
  LEFT JOIN items i ON i.id = pl.item_id
  " . ($hasCategories ? "LEFT JOIN categories c ON c.id = i.category_id" : "") . "
  " . ($hasInvoicesTable ? "LEFT JOIN purchase_invoices pi ON pi.purchase_id = p.id" : "") . "
  WHERE $where
  GROUP BY p.id
  ORDER BY p.id DESC
  LIMIT $perPage OFFSET $offset
";
$listStmt = $pdo->prepare($sql);
$listStmt->execute($params);
$rows = $listStmt->fetchAll();

include __DIR__ . '/../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">Purchases</h3>
    <div class="text-muted">
      Currency: <strong>USD ($)</strong>
      <?php if (table_exists($pdo,'vat_rates')): ?>
        • Current VAT: <strong><?=h(number_format($vatPercentCurrent,2))?>%</strong>
      <?php endif; ?>
    </div>
  </div>

  <div class="d-flex gap-2 align-items-center">
    <a class="btn btn-sm btn-success" href="<?=h(base_url('admin/purchases_new.php'))?>">+ New Purchase</a>

    <form method="get" class="d-flex gap-2 align-items-center">
      <input class="form-control form-control-sm" style="width:260px;"
             name="q" value="<?=h($q)?>" placeholder="Search ID / vendor / invoice / user">

      <select class="form-select form-select-sm" name="per_page" style="width:140px;" onchange="this.form.submit()">
        <?php foreach ([5,10,20,50] as $opt): ?>
          <option value="<?=h($opt)?>" <?=($perPage===$opt?'selected':'')?>><?=h($opt)?> / page</option>
        <?php endforeach; ?>
      </select>

      <input type="hidden" name="page" value="1">
      <button class="btn btn-sm btn-primary">Apply</button>

      <?php if ($q !== ''): ?>
        <a class="btn btn-sm btn-outline-secondary" href="<?=h(base_url('admin/purchases.php'))?>">Clear</a>
      <?php endif; ?>
    </form>
  </div>
</div>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="width:85px;">ID</th>
            <th style="width:170px;">Purchased</th>
            <th style="width:220px;">Vendor</th>
            <th style="width:140px;">Invoice</th>
            <?php if ($hasInvoiceDate): ?><th style="width:140px;">Invoice date</th><?php endif; ?>
            <th style="width:130px;">Files</th>
            <th>Items</th>
            <th style="width:90px;">Qty</th>
            <th style="width:135px;">Subtotal</th>
            <th style="width:110px;">VAT</th>
            <th style="width:135px;">Total</th>
            <th style="width:170px;">By</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="<?=h($hasInvoiceDate?12:11)?>" class="p-4 text-center text-muted">No purchases found.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <?php
              $id = (int)$r['id'];
              $date = $r['purchased_at'] ?: $r['purchase_date'];
              $vendor = $r['vendor_display'] ?: '—';
              $invNo = $r['invoice_no'] ?: '—';
              $invDate = $r['invoice_date'] ?? null;

              $subtotal = (float)$r['subtotal'];
              $vatTotal = (float)$r['vat_total'];
              $grand = (float)$r['grand_total'];

              $vatIncluded = (int)$r['vat_included'] === 1;
              $vatPercent = (float)$r['vat_percent'];

              $items = $r['items_summary'] ?? '';
              $itemsShort = $items;
              if (strlen($itemsShort) > 140) $itemsShort = substr($itemsShort, 0, 140) . '...';

              $files = [];
              $raw = $r['invoice_files'] ?? '';
              if ($raw !== '') {
                foreach (explode('||', $raw) as $chunk) {
                  if (trim($chunk) === '') continue;
                  $parts = explode('::', $chunk, 2);
                  $path = $parts[0] ?? '';
                  $name = $parts[1] ?? '';
                  if ($path !== '') $files[] = ['path'=>$path, 'name'=>$name ?: basename($path)];
                }
              }
            ?>
            <tr>
              <td class="fw-semibold">
                <a href="<?=h(base_url('admin/purchase_view.php?id=' . $id))?>" class="text-decoration-none">
                  #<?=h($id)?>
                </a>
              </td>
              <td class="text-muted">
                <?=h($date)?>
                <div class="small mt-1">
                  <?php if ($vatPercent > 0): ?>
                    <span class="badge bg-secondary"><?=h(number_format($vatPercent,2))?>% VAT</span>
                  <?php endif; ?>
                  <?php if ($vatIncluded): ?>
                    <span class="badge bg-info text-dark">Prices incl. VAT</span>
                  <?php else: ?>
                    <span class="badge bg-light text-dark border">Prices excl. VAT</span>
                  <?php endif; ?>
                </div>
              </td>
              <td class="fw-semibold"><?=h($vendor)?></td>
              <td class="text-muted"><?=h($invNo)?></td>

              <?php if ($hasInvoiceDate): ?>
                <td class="text-muted"><?=h($invDate ?: '—')?></td>
              <?php endif; ?>

              <td>
                <?php if (count($files) === 0): ?>
                  <span class="text-muted">—</span>
                <?php else: ?>
                  <div class="dropdown">
                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                      📎 <?=h(count($files))?>
                    </button>
                    <ul class="dropdown-menu">
                      <?php foreach ($files as $fx): ?>
                        <li>
                          <a class="dropdown-item" href="<?=h(base_url($fx['path']))?>" target="_blank">
                            <?=h($fx['name'])?>
                          </a>
                        </li>
                      <?php endforeach; ?>
                    </ul>
                  </div>
                <?php endif; ?>
              </td>

              <td>
                <div class="fw-semibold"><?=h($itemsShort ?: '—')?></div>
                <?php if ($items !== '' && strlen($items) > 140): ?>
                  <button class="btn btn-link btn-sm p-0" type="button" data-bs-toggle="collapse" data-bs-target="#items<?=h($id)?>">
                    View full list
                  </button>
                  <div class="collapse mt-2" id="items<?=h($id)?>">
                    <div class="small text-muted"><?=h($items)?></div>
                  </div>
                <?php endif; ?>
              </td>

              <td class="fw-semibold"><?=h((int)$r['total_qty'])?></td>
              <td class="text-muted">$<?=h(number_format($subtotal, 2))?></td>
              <td class="text-muted">$<?=h(number_format($vatTotal, 2))?></td>
              <td class="fw-semibold">$<?=h(number_format($grand, 2))?></td>
              <td class="text-muted"><?=h($r['created_by_name'] ?: '—')?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php if ($totalPages > 1): ?>
  <nav class="mt-3">
    <ul class="pagination justify-content-center">
      <?php $prevDisabled = ($page <= 1) ? 'disabled' : ''; ?>
      <?php $nextDisabled = ($page >= $totalPages) ? 'disabled' : ''; ?>

      <li class="page-item <?=h($prevDisabled)?>">
        <a class="page-link" href="?<?=h(build_query_string(['page' => max(1, $page - 1)]))?>">Prev</a>
      </li>

      <?php
        $start = max(1, $page - 2);
        $end = min($totalPages, $page + 2);

        if ($start > 1) {
          echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>1])) . '">1</a></li>';
          if ($start > 2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
        }

        for ($p = $start; $p <= $end; $p++) {
          $active = ($p === $page) ? 'active' : '';
          echo '<li class="page-item ' . h($active) . '"><a class="page-link" href="?' . h(build_query_string(['page'=>$p])) . '">' . h($p) . '</a></li>';
        }

        if ($end < $totalPages) {
          if ($end < $totalPages - 1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
          echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>$totalPages])) . '">' . h($totalPages) . '</a></li>';
        }
      ?>

      <li class="page-item <?=h($nextDisabled)?>">
        <a class="page-link" href="?<?=h(build_query_string(['page' => min($totalPages, $page + 1)]))?>">Next</a>
      </li>
    </ul>

    <div class="text-center text-muted small">
      Showing page <?=h($page)?> of <?=h($totalPages)?> — Total <?=h($total)?> purchases — <?=h($perPage)?> per page
    </div>
  </nav>
<?php endif; ?>

<?php include __DIR__ . '/../partials/footer.php'; ?>
