<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['ADMIN','STOREKEEPER']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

$hasVendors = table_exists($pdo, 'vendors') && column_exists($pdo, 'purchases', 'vendor_id');
$hasInvoiceDate = column_exists($pdo, 'purchases', 'invoice_date');
$hasVatRates = table_exists($pdo, 'vat_rates');
$hasCategories = table_exists($pdo, 'categories') && column_exists($pdo, 'items', 'category_id');

$vatPercent = 0.0;
if ($hasVatRates) {
  $vatPercent = (float)($pdo->query("SELECT vat_percent FROM vat_rates WHERE effective_from <= CURDATE() ORDER BY effective_from DESC LIMIT 1")->fetchColumn() ?: 0);
}

if (!$hasVendors) {
  include __DIR__ . '/../partials/header.php';
  echo '<div class="alert alert-warning shadow-sm"><strong>Vendors are required.</strong><br>'
     . 'Please run the SQL upgrade in <code>sql/upgrade_purchase_vendor_invoice_multi.sql</code> then create vendors in <a href="' . h(base_url('admin/vendors.php')) . '">Vendors</a>.</div>';
  include __DIR__ . '/../partials/footer.php';
  exit;
}

$vendors = $pdo->query("SELECT id, name FROM vendors ORDER BY name")->fetchAll();

$itemsSql = "
  SELECT i.id, i.name, i.unit, i.stock_qty, i.is_active
  " . ($hasCategories ? ", c.name AS category_name" : "") . "
  FROM items i
  " . ($hasCategories ? "LEFT JOIN categories c ON c.id=i.category_id" : "") . "
  WHERE i.is_active=1
  ORDER BY " . ($hasCategories ? "c.name, " : "") . " i.name
";
$items = $pdo->query($itemsSql)->fetchAll();

include __DIR__ . '/../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">New Purchase</h3>
    <div class="text-muted">
      VAT is calculated per item. Current VAT: <strong><?=h(number_format($vatPercent,2))?>%</strong>
    </div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-secondary" href="<?=h(base_url('admin/vendors.php'))?>">Manage Vendors</a>
  </div>
</div>

<form method="post" action="<?=h(base_url('admin/actions/save_purchase.php'))?>" enctype="multipart/form-data" class="card shadow-sm">
  <div class="card-body">

    <div class="row g-3 mb-3">
      <div class="col-md-4">
        <label class="form-label">Vendor</label>
        <select class="form-select" name="vendor_id" required>
          <option value="">— Select vendor —</option>
          <?php foreach ($vendors as $v): ?>
            <option value="<?=h((int)$v['id'])?>"><?=h($v['name'])?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label">Invoice No.</label>
        <input class="form-control" name="invoice_no" placeholder="e.g. INV-2026-001">
      </div>

      <?php if ($hasInvoiceDate): ?>
        <div class="col-md-3">
          <label class="form-label">Invoice date</label>
          <input class="form-control" type="date" name="invoice_date">
        </div>
      <?php endif; ?>

      <div class="col-md-4">
        <label class="form-label">Invoice files (multiple)</label>
        <input class="form-control" type="file" name="invoice_files[]" multiple accept=".pdf,.jpg,.jpeg,.png">
        <div class="form-text">PDF/JPG/PNG • up to 8MB each</div>
      </div>

      <div class="col-md-3">
        <div class="form-check mt-4">
          <input class="form-check-input" type="checkbox" id="vatIncluded" name="vat_included" value="1">
          <label class="form-check-label" for="vatIncluded">
            Prices include VAT
          </label>
        </div>
        <div class="text-muted small">If checked, unit price you enter is VAT-inclusive (for VAT-applied items).</div>
      </div>
    </div>

    <div class="table-responsive">
      <table class="table table-bordered align-middle" id="linesTable">
        <thead class="table-light">
          <tr>
            <th style="width:380px;">Item</th>
            <th style="width:110px;">Qty</th>
            <th style="width:190px;">Unit price (USD)</th>
            <th style="width:120px;" class="text-center">Apply VAT</th>
            <th style="width:140px;">VAT</th>
            <th style="width:180px;">Line total</th>
            <th style="width:80px;"></th>
          </tr>
        </thead>
        <tbody id="linesBody">
          <tr class="line">
            <td>
              <select class="form-select item_id" name="item_id[]" required>
                <option value="">— Select item —</option>
                <?php foreach ($items as $it): ?>
                  <option value="<?=h((int)$it['id'])?>">
                    <?=h(($it['category_name'] ?? '') ? ($it['category_name'].' — ') : '')?><?=h($it['name'])?> (<?=h($it['unit'])?>)
                  </option>
                <?php endforeach; ?>
              </select>
            </td>
            <td><input class="form-control qty" type="number" name="qty[]" min="1" value="1" required></td>
            <td>
              <div class="input-group">
                <span class="input-group-text">$</span>
                <input class="form-control unit_cost" type="number" name="unit_cost[]" min="0" step="0.01" value="0.00" required>
              </div>
            </td>
            <td class="text-center">
              <input class="form-check-input apply_vat" type="checkbox" name="apply_vat[]" value="1" checked>
            </td>
            <td class="vat_cell text-muted">$0.00</td>
            <td class="total_cell fw-semibold">$0.00</td>
            <td class="text-center"><button type="button" class="btn btn-sm btn-outline-danger delRow">✕</button></td>
          </tr>
        </tbody>
        <tfoot class="table-light">
          <tr>
            <th colspan="4" class="text-end">Subtotal</th>
            <th colspan="3" class="text-end" id="subTotal">$0.00</th>
          </tr>
          <tr>
            <th colspan="4" class="text-end">VAT Total</th>
            <th colspan="3" class="text-end" id="vatTotal">$0.00</th>
          </tr>
          <tr>
            <th colspan="4" class="text-end">Grand Total</th>
            <th colspan="3" class="text-end fw-semibold" id="grandTotal">$0.00</th>
          </tr>
        </tfoot>
      </table>
    </div>

    <div class="d-flex gap-2">
      <button type="button" class="btn btn-outline-primary" id="addRow">+ Add line</button>
      <button type="submit" class="btn btn-success ms-auto">Save Purchase</button>
    </div>

    <input type="hidden" id="vatPercent" value="<?=h($vatPercent)?>">
  </div>
</form>

<script>
(function(){
  const vatPercentGlobal = parseFloat(document.getElementById('vatPercent').value || '0');
  const body = document.getElementById('linesBody');
  const addBtn = document.getElementById('addRow');
  const vatIncluded = document.getElementById('vatIncluded');
  const subTotalEl = document.getElementById('subTotal');
  const vatTotalEl = document.getElementById('vatTotal');
  const grandTotalEl = document.getElementById('grandTotal');

  function money(n){ return '$' + (Math.round((n + Number.EPSILON) * 100) / 100).toFixed(2); }

  function recalc(){
    let sub = 0, vatSum = 0, grand = 0;
    const incl = vatIncluded && vatIncluded.checked;

    body.querySelectorAll('tr.line').forEach(tr => {
      const qty = parseFloat(tr.querySelector('.qty').value || '0');
      const priceInput = parseFloat(tr.querySelector('.unit_cost').value || '0');
      const applyVat = tr.querySelector('.apply_vat')?.checked ?? true;
      const vatPercent = applyVat ? vatPercentGlobal : 0;

      let unitNet = priceInput;
      if (applyVat && incl && vatPercent > 0) {
        unitNet = priceInput / (1 + (vatPercent/100));
      }

      const lineSub = qty * unitNet;
      const lineVat = lineSub * (vatPercent/100);
      const lineTotal = lineSub + lineVat;

      sub += lineSub;
      vatSum += lineVat;
      grand += lineTotal;

      tr.querySelector('.vat_cell').textContent = money(lineVat);
      tr.querySelector('.total_cell').textContent = money(lineTotal);
    });

    subTotalEl.textContent = money(sub);
    vatTotalEl.textContent = money(vatSum);
    grandTotalEl.textContent = money(grand);
  }

  function wireRow(tr){
    tr.querySelectorAll('input,select').forEach(el => {
      el.addEventListener('input', recalc);
      el.addEventListener('change', recalc);
    });
    tr.querySelector('.delRow').addEventListener('click', () => {
      const rows = body.querySelectorAll('tr.line');
      if (rows.length <= 1) return;
      tr.remove();
      recalc();
    });
  }

  addBtn.addEventListener('click', () => {
    const first = body.querySelector('tr.line');
    const clone = first.cloneNode(true);
    clone.querySelector('.item_id').value = '';
    clone.querySelector('.qty').value = 1;
    clone.querySelector('.unit_cost').value = '0.00';
    clone.querySelector('.apply_vat').checked = true;
    clone.querySelector('.vat_cell').textContent = '$0.00';
    clone.querySelector('.total_cell').textContent = '$0.00';
    body.appendChild(clone);
    wireRow(clone);
    recalc();
  });

  if (vatIncluded) vatIncluded.addEventListener('change', recalc);

  wireRow(body.querySelector('tr.line'));
  recalc();
})();
</script>

<?php include __DIR__ . '/../partials/footer.php'; ?>
