<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_role(['ADMIN']);
$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function col_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}
function q($k,$d=null){ return $_GET[$k] ?? $d; }
function date_or_default($v,$d){
  if(!$v) return $d;
  $dt = DateTime::createFromFormat('Y-m-d',$v);
  return ($dt && $dt->format('Y-m-d')===$v) ? $v : $d;
}
function clamp_int($v,$min,$max,$d){
  if($v===null || $v==='') return $d;
  $v=(int)$v;
  if($v<$min) return $min;
  if($v>$max) return $max;
  return $v;
}
function fmt_money($v){ return '$'.number_format((float)$v, 2); }

if (!table_exists($pdo,'purchases') || !table_exists($pdo,'purchase_lines') || !table_exists($pdo,'items')) {
  $page_title="Price History";
  include __DIR__ . '/../../partials/header.php';
  echo '<div class="alert alert-danger shadow-sm">Missing tables. Need: <code>purchases</code>, <code>purchase_lines</code>, <code>items</code>.</div>';
  include __DIR__ . '/../../partials/footer.php';
  exit;
}

$hasVendors = table_exists($pdo,'vendors') && col_exists($pdo,'purchases','vendor_id');
$hasVendorNameCol = col_exists($pdo,'purchases','vendor_name') || col_exists($pdo,'purchases','vendor');

// Build SAFE date expression that works even when invoice_date is NULL
$parts = [];
if (col_exists($pdo,'purchases','invoice_date'))   $parts[] = "p.invoice_date";
if (col_exists($pdo,'purchases','purchase_date'))  $parts[] = "p.purchase_date";
if (col_exists($pdo,'purchases','purchased_at'))   $parts[] = "DATE(p.purchased_at)";
if (col_exists($pdo,'purchases','created_at'))     $parts[] = "DATE(p.created_at)";
if (!$parts) $parts[] = "CURDATE()";
$dateExpr = "COALESCE(" . implode(", ", $parts) . ")";

$vendorExpr = "'—'";
if ($hasVendors) $vendorExpr = "COALESCE(v.name,'—')";
elseif ($hasVendorNameCol) $vendorExpr = "COALESCE(p.vendor_name, p.vendor, '—')";

$today = date('Y-m-d');
// IMPORTANT: default range is wide so you don't miss old purchases
$from = date_or_default(q('from'), '2000-01-01');
$to   = date_or_default(q('to'), $today);

$item_id = (int)q('item_id', 0);
$search = trim((string)q('search',''));

$page = clamp_int(q('page'), 1, 1000000, 1);
$per_page = clamp_int(q('per_page'), 5, 200, 25);
$offset = ($page-1)*$per_page;

// Items dropdown
$items = $pdo->query("SELECT id, name FROM items ORDER BY name")->fetchAll();

$where = [];
$params = [];
$where[] = "$dateExpr BETWEEN ? AND ?";
$params[] = $from; $params[] = $to;

if ($item_id > 0) { $where[] = "pl.item_id = ?"; $params[] = $item_id; }
if ($search !== '') {
  $where[] = "(i.name LIKE ? OR COALESCE(p.invoice_no,'') LIKE ? OR $vendorExpr LIKE ?)";
  $like = "%{$search}%";
  $params[] = $like; $params[] = $like; $params[] = $like;
}
$whereSql = "WHERE " . implode(" AND ", $where);

$baseSql = "
  FROM purchase_lines pl
  JOIN purchases p ON p.id = pl.purchase_id
  JOIN items i ON i.id = pl.item_id
  ".($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "")."
  $whereSql
";

// Summary
$sumStmt = $pdo->prepare("
  SELECT MIN(pl.unit_cost) AS min_price,
         MAX(pl.unit_cost) AS max_price,
         AVG(pl.unit_cost) AS avg_price
  $baseSql
");
$sumStmt->execute($params);
$summary = $sumStmt->fetch() ?: ['min_price'=>0,'max_price'=>0,'avg_price'=>0];

$lastStmt = $pdo->prepare("
  SELECT pl.unit_cost AS last_price, $dateExpr AS last_date
  $baseSql
  ORDER BY $dateExpr DESC, p.id DESC, pl.id DESC
  LIMIT 1
");
$lastStmt->execute($params);
$last = $lastStmt->fetch();
$last_price = $last['last_price'] ?? 0;
$last_date = $last['last_date'] ?? '—';

$countStmt = $pdo->prepare("SELECT COUNT(*) $baseSql");
$countStmt->execute($params);
$total_rows = (int)$countStmt->fetchColumn();
$total_pages = max(1, (int)ceil($total_rows / $per_page));
if ($page > $total_pages) { $page = $total_pages; $offset = ($page-1)*$per_page; }

// Fetch rows
$rows = [];
if ($item_id > 0) {
  // Fetch ALL rows for this item to compute change% correctly (even across pages)
  $stmt = $pdo->prepare("
    SELECT
      p.id AS purchase_id,
      $dateExpr AS price_date,
      $vendorExpr AS vendor_name,
      COALESCE(p.invoice_no,'') AS invoice_no,
      i.id AS item_id,
      i.name AS item_name,
      pl.id AS line_id,
      pl.qty,
      pl.unit_cost,
      (pl.qty * pl.unit_cost) AS line_total
    $baseSql
    ORDER BY price_date ASC, p.id ASC, pl.id ASC
  ");
  $stmt->execute($params);
  $all = $stmt->fetchAll();

  $prev = null;
  foreach ($all as &$r) {
    $r['change_pct'] = null;
    if ($prev !== null && (float)$prev > 0) {
      $r['change_pct'] = (((float)$r['unit_cost'] - (float)$prev) / (float)$prev) * 100.0;
    }
    $prev = (float)$r['unit_cost'];
  }
  unset($r);

  $rows = array_slice($all, $offset, $per_page);
} else {
  $stmt = $pdo->prepare("
    SELECT
      p.id AS purchase_id,
      $dateExpr AS price_date,
      $vendorExpr AS vendor_name,
      COALESCE(p.invoice_no,'') AS invoice_no,
      i.id AS item_id,
      i.name AS item_name,
      pl.id AS line_id,
      pl.qty,
      pl.unit_cost,
      (pl.qty * pl.unit_cost) AS line_total,
      NULL AS change_pct
    $baseSql
    ORDER BY price_date DESC, p.id DESC, pl.id DESC
    LIMIT $per_page OFFSET $offset
  ");
  $stmt->execute($params);
  $rows = $stmt->fetchAll();
}

// Group by purchase_id (within page)
$groups = [];
foreach ($rows as $r) {
  $pid = (int)$r['purchase_id'];
  if (!isset($groups[$pid])) {
    $groups[$pid] = [
      'purchase_id' => $pid,
      'price_date' => $r['price_date'],
      'vendor_name' => $r['vendor_name'],
      'invoice_no' => $r['invoice_no'],
      'lines' => []
    ];
  }
  $groups[$pid]['lines'][] = $r;
}
$groups_list = array_values($groups);
usort($groups_list, function($a,$b){
  if ($a['price_date'] === $b['price_date']) return $b['purchase_id'] <=> $a['purchase_id'];
  return strcmp($b['price_date'], $a['price_date']);
});

function qs_keep($overrides=[]) {
  $base = $_GET;
  foreach ($overrides as $k=>$v) $base[$k] = $v;
  return http_build_query($base);
}

$page_title = "Price History";
include __DIR__ . '/../../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">Product Purchase Activity (Price History)</h3>
    <div class="text-muted">Shows all purchases (qty + vendor + unit price) and % change vs previous</div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-secondary" href="<?=h(base_url('admin/reports/price_history_export.php?'.qs_keep()))?>">
      <i class="bi bi-file-earmark-excel me-1"></i> Export Excel (CSV)
    </a>
  </div>
</div>

<form class="card shadow-sm mb-3" method="get">
  <div class="card-body">
    <div class="row g-2 align-items-end">
      <div class="col-md-4">
        <label class="form-label">Product</label>
        <select class="form-select" name="item_id">
          <option value="0">All products</option>
          <?php foreach ($items as $it): ?>
            <option value="<?=h((int)$it['id'])?>" <?=($item_id===(int)$it['id']?'selected':'')?>>
              <?=h($it['name'])?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">To get % change, select one product.</div>
      </div>

      <div class="col-md-2">
        <label class="form-label">From</label>
        <input class="form-control" type="date" name="from" value="<?=h($from)?>">
      </div>

      <div class="col-md-2">
        <label class="form-label">To</label>
        <input class="form-control" type="date" name="to" value="<?=h($to)?>">
      </div>

      <div class="col-md-4">
        <label class="form-label">Search</label>
        <input class="form-control" name="search" value="<?=h($search)?>" placeholder="Invoice, vendor, item…">
      </div>

      <div class="col-md-2">
        <label class="form-label">Rows per page</label>
        <select class="form-select" name="per_page">
          <?php foreach ([10,25,50,100,200] as $n): ?>
            <option value="<?=h($n)?>" <?=($per_page===$n?'selected':'')?>><?=h($n)?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2">
        <button class="btn btn-primary w-100"><i class="bi bi-funnel me-1"></i> Apply</button>
      </div>

      <div class="col-md-2">
        <a class="btn btn-outline-secondary w-100" href="<?=h(base_url('admin/reports/price_history.php'))?>">Reset</a>
      </div>
    </div>
  </div>
</form>

<div class="row g-3 mb-3">
  <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
    <div class="text-muted small">Min Unit Price</div><div class="fs-4 fw-bold"><?=h(fmt_money($summary['min_price']))?></div>
  </div></div></div>
  <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
    <div class="text-muted small">Max Unit Price</div><div class="fs-4 fw-bold"><?=h(fmt_money($summary['max_price']))?></div>
  </div></div></div>
  <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
    <div class="text-muted small">Average Unit Price</div><div class="fs-4 fw-bold"><?=h(fmt_money($summary['avg_price']))?></div>
  </div></div></div>
  <div class="col-md-3"><div class="card shadow-sm"><div class="card-body">
    <div class="text-muted small">Last Price</div><div class="fs-4 fw-bold"><?=h(fmt_money($last_price))?></div>
    <div class="text-muted small">Date: <?=h($last_date)?></div>
  </div></div></div>
</div>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="p-3 border-bottom d-flex justify-content-between align-items-center">
      <div>
        <h5 class="mb-0">Purchases (grouped)</h5>
        <div class="text-muted small"><?=h($total_rows)?> line(s) • Page <?=h($page)?> / <?=h($total_pages)?></div>
      </div>

      <nav aria-label="Pagination">
        <ul class="pagination pagination-sm mb-0">
          <li class="page-item <?=($page<=1?'disabled':'')?>">
            <a class="page-link" href="?<?=h(qs_keep(['page'=>max(1,$page-1)]))?>">Prev</a>
          </li>
          <li class="page-item disabled"><span class="page-link"><?=h($page)?></span></li>
          <li class="page-item <?=($page>=$total_pages?'disabled':'')?>">
            <a class="page-link" href="?<?=h(qs_keep(['page'=>min($total_pages,$page+1)]))?>">Next</a>
          </li>
        </ul>
      </nav>
    </div>

    <?php if (!$groups_list): ?>
      <div class="p-4 text-center text-muted">No rows found.</div>
    <?php endif; ?>

    <?php foreach ($groups_list as $g): ?>
      <div class="p-3 border-bottom bg-light d-flex flex-wrap gap-2 justify-content-between align-items-center">
        <div class="d-flex flex-wrap gap-3 align-items-center">
          <span class="badge bg-dark">Purchase #<?=h($g['purchase_id'])?></span>
          <span class="text-muted"><i class="bi bi-calendar3 me-1"></i><?=h($g['price_date'])?></span>
          <span class="text-muted"><i class="bi bi-buildings me-1"></i><?=h($g['vendor_name'] ?: '—')?></span>
          <span class="text-muted"><i class="bi bi-receipt me-1"></i><?=h($g['invoice_no'] ?: '—')?></span>
        </div>
        <div>
          <a class="btn btn-sm btn-outline-primary" href="<?=h(base_url('admin/purchase_view.php?id='.(int)$g['purchase_id']))?>">View Purchase</a>
        </div>
      </div>

      <div class="table-responsive">
        <table class="table table-hover align-middle mb-0">
          <thead class="table-light">
            <tr>
              <th>Item</th>
              <th class="text-end" style="width:110px;">Qty</th>
              <th class="text-end" style="width:150px;">Unit Price</th>
              <th class="text-end" style="width:170px;">Line Total</th>
              <th class="text-end" style="width:180px;">Change vs previous</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($g['lines'] as $r): ?>
              <?php
                $cp = $r['change_pct'];
                $changeHtml = '—';
                if ($item_id > 0) {
                  if ($cp === null) $changeHtml = '—';
                  else {
                    $num = number_format((float)$cp, 2);
                    if ((float)$cp > 0) $changeHtml = '<span class="text-success fw-semibold">▲ '.$num.'%</span>';
                    elseif ((float)$cp < 0) $changeHtml = '<span class="text-danger fw-semibold">▼ '.$num.'%</span>';
                    else $changeHtml = '<span class="text-muted fw-semibold">0.00%</span>';
                  }
                }
              ?>
              <tr>
                <td class="fw-semibold"><?=h($r['item_name'])?></td>
                <td class="text-end"><?=h((int)$r['qty'])?></td>
                <td class="text-end fw-semibold"><?=h(fmt_money($r['unit_cost']))?></td>
                <td class="text-end"><?=h(fmt_money($r['line_total']))?></td>
                <td class="text-end"><?=$changeHtml?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    <?php endforeach; ?>
  </div>
</div>


<?php include __DIR__ . '/../../partials/footer.php'; ?>
