<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_role(['ADMIN']);

$pdo = db();

function q(string $k, $default=null) { return $_GET[$k] ?? $default; }
function clamp_int($v, $min, $max, $default) {
  if ($v === null || $v === '') return $default;
  $v = (int)$v;
  if ($v < $min) return $min;
  if ($v > $max) return $max;
  return $v;
}
function date_or_default($v, $default) {
  if (!$v) return $default;
  $dt = DateTime::createFromFormat('Y-m-d', $v);
  return ($dt && $dt->format('Y-m-d') === $v) ? $v : $default;
}

$today = date('Y-m-d');
$from = date_or_default(q('from'), date('Y-m-01'));
$to   = date_or_default(q('to'), $today);
$status = trim((string)q('status',''));
$requester_id = (int)q('requester_id', 0);
$search = trim((string)q('search',''));

$page = clamp_int(q('page'), 1, 1000000, 1);
$per_page = clamp_int(q('per_page'), 5, 100, 10);
$offset = ($page-1)*$per_page;

// Requesters dropdown
$requesters = $pdo->query("SELECT id, full_name, email FROM users WHERE role='REQUESTER' ORDER BY full_name")->fetchAll();

// WHERE
$where = [];
$params = [];

$where[] = "DATE(r.created_at) BETWEEN ? AND ?";
$params[] = $from;
$params[] = $to;

if ($status !== '') { $where[] = "r.status = ?"; $params[] = $status; }
if ($requester_id > 0) { $where[] = "r.requester_id = ?"; $params[] = $requester_id; }
if ($search !== '') {
  $where[] = "(CAST(r.id AS CHAR) LIKE ? OR u.full_name LIKE ? OR i.name LIKE ? OR COALESCE(sk.full_name,'') LIKE ?)";
  $like = "%{$search}%";
  $params[] = $like; $params[] = $like; $params[] = $like; $params[] = $like;
}
$whereSql = $where ? ("WHERE " . implode(" AND ", $where)) : "";

// Summary cards
$summaryStmt = $pdo->prepare("
  SELECT
    COUNT(DISTINCT r.id) AS total_requests,
    COALESCE(SUM(rl.qty_requested),0) AS total_qty_requested,
    COALESCE(SUM(rl.qty_delivered),0) AS total_qty_delivered
  FROM requests r
  JOIN users u ON u.id = r.requester_id
  JOIN request_lines rl ON rl.request_id = r.id
  JOIN items i ON i.id = rl.item_id
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  $whereSql
");
$summaryStmt->execute($params);
$summary = $summaryStmt->fetch() ?: ['total_requests'=>0,'total_qty_requested'=>0,'total_qty_delivered'=>0];

// Top requested item
$topItemStmt = $pdo->prepare("
  SELECT i.name, COALESCE(SUM(rl.qty_requested),0) AS qty
  FROM requests r
  JOIN users u ON u.id = r.requester_id
  JOIN request_lines rl ON rl.request_id = r.id
  JOIN items i ON i.id = rl.item_id
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  $whereSql
  GROUP BY i.id, i.name
  ORDER BY qty DESC, i.name ASC
  LIMIT 1
");
$topItemStmt->execute($params);
$topItem = $topItemStmt->fetch();

// Grouped summary by requester (top 10)
$groupStmt = $pdo->prepare("
  SELECT
    u.id AS requester_id,
    u.full_name AS requester_name,
    COUNT(DISTINCT r.id) AS requests_count,
    COALESCE(SUM(rl.qty_requested),0) AS qty_requested,
    COALESCE(SUM(rl.qty_delivered),0) AS qty_delivered
  FROM requests r
  JOIN users u ON u.id = r.requester_id
  JOIN request_lines rl ON rl.request_id = r.id
  JOIN items i ON i.id = rl.item_id
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  $whereSql
  GROUP BY u.id, u.full_name
  ORDER BY requests_count DESC, qty_requested DESC, u.full_name ASC
  LIMIT 10
");
$groupStmt->execute($params);
$topRequesters = $groupStmt->fetchAll();

// Count for pagination (detail rows)
$countStmt = $pdo->prepare("
  SELECT COUNT(*) FROM (
    SELECT rl.id
    FROM requests r
    JOIN users u ON u.id = r.requester_id
    JOIN request_lines rl ON rl.request_id = r.id
    JOIN items i ON i.id = rl.item_id
    LEFT JOIN users sk ON sk.id = r.assigned_to_id
    $whereSql
  ) x
");
$countStmt->execute($params);
$total_rows = (int)$countStmt->fetchColumn();
$total_pages = max(1, (int)ceil($total_rows / $per_page));
if ($page > $total_pages) { $page = $total_pages; $offset = ($page-1)*$per_page; }

// Detail rows
$detailStmt = $pdo->prepare("
  SELECT
    r.id AS request_id,
    r.created_at,
    r.status,
    u.full_name AS requester_name,
    COALESCE(sk.full_name,'—') AS storekeeper_name,
    i.name AS item_name,
    rl.qty_requested,
    rl.qty_delivered
  FROM requests r
  JOIN users u ON u.id = r.requester_id
  JOIN request_lines rl ON rl.request_id = r.id
  JOIN items i ON i.id = rl.item_id
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  $whereSql
  ORDER BY r.created_at DESC, r.id DESC
  LIMIT $per_page OFFSET $offset
");
$detailStmt->execute($params);
$rows = $detailStmt->fetchAll();

$statuses = ['SUBMITTED','PREPARING','DELIVERED','RECEIVED','CANCELLED'];

$page_title = "Requester Report";
include __DIR__ . '/../../partials/header.php';

function badge_class($st) {
  $st = strtoupper((string)$st);
  if ($st === 'RECEIVED') return 'bg-success';
  if ($st === 'DELIVERED') return 'bg-primary';
  if ($st === 'PREPARING') return 'bg-warning text-dark';
  if ($st === 'SUBMITTED') return 'bg-danger';
  return 'bg-secondary';
}
function qs_keep($overrides=[]) {
  $base = $_GET;
  foreach ($overrides as $k=>$v) { $base[$k] = $v; }
  return http_build_query($base);
}
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">Report by Requester</h3>
    <div class="text-muted">Filter and audit consumption requests • Export to Excel</div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-outline-secondary" href="<?=h(base_url('admin/reports/requester_report_export.php?'.qs_keep()))?>">
      <i class="bi bi-file-earmark-excel me-1"></i> Export Excel (CSV)
    </a>
  </div>
</div>

<form class="card shadow-sm mb-3" method="get">
  <div class="card-body">
    <div class="row g-2 align-items-end">
      <div class="col-md-3">
        <label class="form-label">Requester</label>
        <select class="form-select" name="requester_id">
          <option value="0">All requesters</option>
          <?php foreach ($requesters as $r): ?>
            <option value="<?=h((int)$r['id'])?>" <?=($requester_id===(int)$r['id']?'selected':'')?>>
              <?=h($r['full_name'])?><?=!empty($r['email'])?' • '.$r['email']:''?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2">
        <label class="form-label">From</label>
        <input class="form-control" type="date" name="from" value="<?=h($from)?>">
      </div>

      <div class="col-md-2">
        <label class="form-label">To</label>
        <input class="form-control" type="date" name="to" value="<?=h($to)?>">
      </div>

      <div class="col-md-2">
        <label class="form-label">Status</label>
        <select class="form-select" name="status">
          <option value="">All</option>
          <?php foreach ($statuses as $st): ?>
            <option value="<?=h($st)?>" <?=($status===$st?'selected':'')?>>
              <?=h($st)?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label">Search</label>
        <input class="form-control" name="search" value="<?=h($search)?>" placeholder="Request #, item, requester, storekeeper…">
      </div>

      <div class="col-md-2">
        <label class="form-label">Rows per page</label>
        <select class="form-select" name="per_page">
          <?php foreach ([5,10,15,25,50,100] as $n): ?>
            <option value="<?=h($n)?>" <?=($per_page===$n?'selected':'')?>><?=h($n)?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2">
        <button class="btn btn-primary w-100">
          <i class="bi bi-funnel me-1"></i> Apply
        </button>
      </div>

      <div class="col-md-2">
        <a class="btn btn-outline-secondary w-100" href="<?=h(base_url('admin/reports/requester_report.php'))?>">
          Reset
        </a>
      </div>
    </div>
  </div>
</form>

<div class="row g-3 mb-3">
  <div class="col-md-3">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="text-muted small">Total Requests</div>
        <div class="fs-3 fw-bold"><?=h((int)$summary['total_requests'])?></div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="text-muted small">Qty Requested</div>
        <div class="fs-3 fw-bold"><?=h((int)$summary['total_qty_requested'])?></div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="text-muted small">Qty Delivered</div>
        <div class="fs-3 fw-bold"><?=h((int)$summary['total_qty_delivered'])?></div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="text-muted small">Top Item</div>
        <div class="fw-bold"><?=h($topItem['name'] ?? '—')?></div>
        <div class="text-muted small"><?=h((int)($topItem['qty'] ?? 0))?> requested</div>
      </div>
    </div>
  </div>
</div>

<div class="card shadow-sm mb-3">
  <div class="card-body p-0">
    <div class="p-3 border-bottom">
      <h5 class="mb-0">Top requesters (top 10)</h5>
      <div class="text-muted small">Within selected filters</div>
    </div>
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th>Requester</th>
            <th class="text-end" style="width:140px;">Requests</th>
            <th class="text-end" style="width:170px;">Qty Requested</th>
            <th class="text-end" style="width:170px;">Qty Delivered</th>
            <th style="width:180px;"></th>
          </tr>
        </thead>
        <tbody>
        <?php if (!$topRequesters): ?>
          <tr><td colspan="5" class="p-4 text-center text-muted">No data in this range.</td></tr>
        <?php endif; ?>
        <?php foreach ($topRequesters as $tr): ?>
          <tr>
            <td class="fw-semibold"><?=h($tr['requester_name'])?></td>
            <td class="text-end fw-semibold"><?=h((int)$tr['requests_count'])?></td>
            <td class="text-end"><?=h((int)$tr['qty_requested'])?></td>
            <td class="text-end"><?=h((int)$tr['qty_delivered'])?></td>
            <td class="text-end">
              <a class="btn btn-sm btn-outline-primary"
                 href="<?=h(base_url('admin/reports/requester_report.php?'.qs_keep(['requester_id'=>(int)$tr['requester_id'], 'page'=>1])))?>">
                View details
              </a>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="p-3 border-bottom d-flex justify-content-between align-items-center">
      <div>
        <h5 class="mb-0">Details</h5>
        <div class="text-muted small"><?=h($total_rows)?> row(s) • Page <?=h($page)?> / <?=h($total_pages)?></div>
      </div>
      <nav aria-label="Pagination">
        <ul class="pagination pagination-sm mb-0">
          <li class="page-item <?=($page<=1?'disabled':'')?>">
            <a class="page-link" href="?<?=h(qs_keep(['page'=>max(1,$page-1)]))?>">Prev</a>
          </li>
          <li class="page-item disabled"><span class="page-link"><?=h($page)?></span></li>
          <li class="page-item <?=($page>=$total_pages?'disabled':'')?>">
            <a class="page-link" href="?<?=h(qs_keep(['page'=>min($total_pages,$page+1)]))?>">Next</a>
          </li>
        </ul>
      </nav>
    </div>

    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="width:110px;">Request #</th>
            <th style="width:170px;">Date</th>
            <th>Requester</th>
            <th>Item</th>
            <th class="text-end" style="width:140px;">Requested</th>
            <th class="text-end" style="width:140px;">Delivered</th>
            <th style="width:130px;">Status</th>
            <th style="width:180px;">Storekeeper</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="8" class="p-4 text-center text-muted">No rows found.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="fw-semibold">
                <a href="<?=h(base_url('admin/request_view.php?id='.(int)$r['request_id']))?>">#<?=h((int)$r['request_id'])?></a>
              </td>
              <td class="text-muted"><?=h($r['created_at'])?></td>
              <td><?=h($r['requester_name'])?></td>
              <td class="fw-semibold"><?=h($r['item_name'])?></td>
              <td class="text-end"><?=h((int)$r['qty_requested'])?></td>
              <td class="text-end"><?=h((int)$r['qty_delivered'])?></td>
              <td><span class="badge <?=h(badge_class($r['status']))?>"><?=h($r['status'])?></span></td>
              <td class="text-muted"><?=h($r['storekeeper_name'])?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../../partials/footer.php'; ?>
