<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_role(['ADMIN']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}
function build_query_string(array $params): string {
  $merged = array_merge($_GET, $params);
  foreach ($merged as $k => $v) {
    if ($v === '' || $v === null) unset($merged[$k]);
  }
  return http_build_query($merged);
}

$hasStockMoves = table_exists($pdo,'stock_moves');
$hasMoveCreatedAt = $hasStockMoves && column_exists($pdo,'stock_moves','created_at');
$hasMoveType = $hasStockMoves && column_exists($pdo,'stock_moves','move_type');
$hasMoveQty = $hasStockMoves && column_exists($pdo,'stock_moves','qty');
$hasMoveItem = $hasStockMoves && column_exists($pdo,'stock_moves','item_id');

if (!$hasStockMoves || !$hasMoveQty || !$hasMoveItem) {
  include __DIR__ . '/../../partials/header.php';
  echo '<div class="alert alert-warning shadow-sm"><strong>stock_moves table is missing or incomplete.</strong><br>'
     . 'This report requires <code>stock_moves(item_id, qty, created_at)</code>.</div>';
  include __DIR__ . '/../../partials/footer.php';
  exit;
}

$hasCategories = table_exists($pdo,'categories') && column_exists($pdo,'items','category_id');
$hasPurchases = table_exists($pdo,'purchases');
$hasVendors = table_exists($pdo,'vendors') && $hasPurchases && column_exists($pdo,'purchases','vendor_id');
$hasRequests = table_exists($pdo,'requests') && table_exists($pdo,'users') && column_exists($pdo,'requests','requester_id');

$itemId = (int)($_GET['item_id'] ?? 0);
$dateFrom = trim($_GET['from'] ?? '');
$dateTo = trim($_GET['to'] ?? '');

$page = max(1, (int)($_GET['page'] ?? 1));
$perOptions = [5,10,15,20,50,100];
$perPage = (int)($_GET['per_page'] ?? 10);
if (!in_array($perPage, $perOptions, true)) $perPage = 10;

if ($dateFrom === '') $dateFrom = date('Y-m-01');
if ($dateTo === '') $dateTo = date('Y-m-d');

$itemsSql = "
  SELECT i.id, i.name, i.unit, i.stock_qty
  " . ($hasCategories ? ", c.name AS category_name" : "") . "
  FROM items i
  " . ($hasCategories ? "LEFT JOIN categories c ON c.id=i.category_id" : "") . "
  ORDER BY " . ($hasCategories ? "c.name, " : "") . " i.name
";
$items = $pdo->query($itemsSql)->fetchAll();

$meta = null;
$rows = [];
$opening = null;
$periodNet = 0.0;
$closing = null;
$totalRows = 0;
$totalPages = 1;

if ($itemId > 0) {
  $metaStmt = $pdo->prepare("
    SELECT i.id, i.name, i.unit, i.stock_qty
      " . ($hasCategories ? ", c.name AS category_name" : "") . "
    FROM items i
    " . ($hasCategories ? "LEFT JOIN categories c ON c.id=i.category_id" : "") . "
    WHERE i.id=?
    LIMIT 1
  ");
  $metaStmt->execute([$itemId]);
  $meta = $metaStmt->fetch();

  if ($meta) {
    $sumSince = $pdo->prepare("
      SELECT COALESCE(SUM(qty),0) AS s
      FROM stock_moves
      WHERE item_id=? AND DATE(" . ($hasMoveCreatedAt ? "created_at" : "NOW()") . ") >= ?
    ");
    $sumSince->execute([$itemId, $dateFrom]);
    $movesSince = (float)($sumSince->fetchColumn() ?: 0);
    $opening = (float)$meta['stock_qty'] - $movesSince;

    $cnt = $pdo->prepare("
      SELECT COUNT(*)
      FROM stock_moves sm
      WHERE sm.item_id=?
        AND DATE(" . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . ") BETWEEN ? AND ?
    ");
    $cnt->execute([$itemId, $dateFrom, $dateTo]);
    $totalRows = (int)($cnt->fetchColumn() ?: 0);
    $totalPages = max(1, (int)ceil($totalRows / $perPage));
    if ($page > $totalPages) $page = $totalPages;
    $offset = ($page - 1) * $perPage;

    $netStmt = $pdo->prepare("
      SELECT COALESCE(SUM(qty),0)
      FROM stock_moves sm
      WHERE sm.item_id=?
        AND DATE(" . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . ") BETWEEN ? AND ?
    ");
    $netStmt->execute([$itemId, $dateFrom, $dateTo]);
    $periodNet = (float)($netStmt->fetchColumn() ?: 0);
    $closing = $opening + $periodNet;

    $mvSql = "
      SELECT
        sm.id,
        " . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . " AS created_at,
        " . ($hasMoveType ? "sm.move_type" : "''") . " AS move_type,
        COALESCE(sm.ref_table,'') AS ref_table,
        COALESCE(sm.ref_id,0) AS ref_id,
        sm.qty,
        COALESCE(sm.note,'') AS note
        " . ($hasVendors ? ",
            CASE WHEN sm.ref_table='purchases' THEN COALESCE(v.name,'') ELSE '' END AS vendor_name
          " : ", '' AS vendor_name") . "
        " . ($hasRequests ? ",
            CASE WHEN sm.ref_table='requests' THEN COALESCE(req.full_name,'') ELSE '' END AS requester_name
          " : ", '' AS requester_name") . "
      FROM stock_moves sm
      " . ($hasPurchases ? "LEFT JOIN purchases p ON (sm.ref_table='purchases' AND p.id=sm.ref_id)" : "") . "
      " . ($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "") . "
      " . ($hasRequests ? "LEFT JOIN requests r ON (sm.ref_table='requests' AND r.id=sm.ref_id)
                           LEFT JOIN users req ON req.id = r.requester_id" : "") . "
      WHERE sm.item_id=?
        AND DATE(" . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . ") BETWEEN ? AND ?
      ORDER BY " . ($hasMoveCreatedAt ? "sm.created_at" : "sm.ref_id") . " ASC, sm.id ASC
      LIMIT $perPage OFFSET $offset
    ";
    $mv = $pdo->prepare($mvSql);
    $mv->execute([$itemId, $dateFrom, $dateTo]);
    $rows = $mv->fetchAll();
  }
}

include __DIR__ . '/../../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">Stock Movement Report</h3>
    <div class="text-muted">By product • Opening stock on top • Smart reference links • Export to Excel</div>
  </div>

  <?php if ($itemId > 0): ?>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-success" href="<?=h(base_url('admin/reports/stock_movement_export.php?' . build_query_string([])))?>">
        ⬇ Export Excel
      </a>
    </div>
  <?php endif; ?>
</div>

<div class="card shadow-sm mb-3">
  <div class="card-body">
    <form method="get" class="row g-2 align-items-end">
      <div class="col-md-5">
        <label class="form-label">Product</label>
        <select class="form-select" name="item_id" required>
          <option value="">— Select product —</option>
          <?php foreach ($items as $it): ?>
            <?php
              $label = ($it['category_name'] ?? '') ? ($it['category_name'] . ' — ' . $it['name']) : $it['name'];
              $label .= ' (' . $it['unit'] . ')';
            ?>
            <option value="<?=h((int)$it['id'])?>" <?=($itemId===(int)$it['id']?'selected':'')?>>
              <?=h($label)?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2">
        <label class="form-label">From</label>
        <input class="form-control" type="date" name="from" value="<?=h($dateFrom)?>" required>
      </div>

      <div class="col-md-2">
        <label class="form-label">To</label>
        <input class="form-control" type="date" name="to" value="<?=h($dateTo)?>" required>
      </div>

      <div class="col-md-1">
        <label class="form-label">Rows</label>
        <select class="form-select" name="per_page" onchange="this.form.submit()">
          <?php foreach ([5,10,15,20,50,100] as $opt): ?>
            <option value="<?=h($opt)?>" <?=($perPage===$opt?'selected':'')?>>
              <?=h($opt)?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2 d-flex gap-2">
        <button class="btn btn-primary w-100">Run</button>
        <a class="btn btn-outline-secondary w-100" href="<?=h(base_url('admin/reports/stock_movement.php'))?>">Reset</a>
      </div>
    </form>
  </div>
</div>

<?php if ($itemId > 0 && !$meta): ?>
  <div class="alert alert-danger shadow-sm">Product not found.</div>
<?php endif; ?>

<?php if ($meta): ?>
  <div class="row g-3 mb-3">
    <div class="col-md-4">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="text-muted small">Product</div>
          <div class="fw-semibold"><?=h(($meta['category_name'] ?? '') ? ($meta['category_name'].' — ') : '')?><?=h($meta['name'])?></div>
          <div class="text-muted small">Unit: <?=h($meta['unit'])?></div>
        </div>
      </div>
    </div>

    <div class="col-md-2">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="text-muted small">Opening stock (<?=h($dateFrom)?>)</div>
          <div class="display-6 mb-0"><?=h(number_format($opening,0))?></div>
        </div>
      </div>
    </div>

    <div class="col-md-2">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="text-muted small">Net movement</div>
          <div class="display-6 mb-0 <?=($periodNet>=0?'text-success':'text-danger')?>"><?=h(($periodNet>=0?'+':'') . number_format($periodNet,0))?></div>
        </div>
      </div>
    </div>

    <div class="col-md-2">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="text-muted small">Closing stock (<?=h($dateTo)?>)</div>
          <div class="display-6 mb-0"><?=h(number_format($closing,0))?></div>
        </div>
      </div>
    </div>

    <div class="col-md-2">
      <div class="card shadow-sm">
        <div class="card-body">
          <div class="text-muted small">Current stock (now)</div>
          <div class="display-6 mb-0"><?=h(number_format((float)$meta['stock_qty'],0))?></div>
        </div>
      </div>
    </div>
  </div>

  <div class="card shadow-sm">
    <div class="card-body p-0">
      <div class="table-responsive">
        <table class="table table-hover align-middle mb-0">
          <thead class="table-light">
            <tr>
              <th style="width:190px;">Date/Time</th>
              <th style="width:140px;">Type</th>
              <th style="width:170px;">Reference</th>
              <th style="width:220px;">Vendor / Requester</th>
              <th style="width:110px;" class="text-end">Qty</th>
              <th>Note</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$rows): ?>
              <tr><td colspan="6" class="p-4 text-center text-muted">No movements in this period.</td></tr>
            <?php endif; ?>

            <?php foreach ($rows as $r): ?>
              <?php
                $qty = (float)$r['qty'];
                $badge = ($qty>=0) ? 'bg-success' : 'bg-danger';
                $refTable = trim($r['ref_table']);
                $refId = (int)$r['ref_id'];
                $refText = ($refTable !== '' && $refId > 0) ? ($refTable . ' #' . $refId) : '—';

                $refUrl = '';
                if ($refTable === 'purchases') $refUrl = base_url('admin/purchase_view.php?id=' . $refId);
                if ($refTable === 'requests')  $refUrl = base_url('admin/request_view.php?id=' . $refId);

                $who = '';
                if (!empty($r['vendor_name'])) $who = $r['vendor_name'];
                if (!empty($r['requester_name'])) $who = $r['requester_name'];
              ?>
              <tr>
                <td class="text-muted"><?=h($r['created_at'])?></td>
                <td>
                  <?php if (!empty($r['move_type'])): ?>
                    <span class="badge bg-secondary"><?=h($r['move_type'])?></span>
                  <?php else: ?>
                    <span class="text-muted">—</span>
                  <?php endif; ?>
                </td>
                <td>
                  <?php if ($refUrl !== '' && $refId > 0): ?>
                    <a href="<?=h($refUrl)?>" class="text-decoration-none fw-semibold"><?=h($refText)?></a>
                  <?php else: ?>
                    <span class="text-muted"><?=h($refText)?></span>
                  <?php endif; ?>
                </td>
                <td class="text-muted"><?=h($who ?: '—')?></td>
                <td class="text-end"><span class="badge <?=h($badge)?>"><?=h(($qty>=0?'+':'') . number_format($qty,0))?></span></td>
                <td><?=h($r['note'])?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

  <?php if ($totalPages > 1): ?>
    <nav class="mt-3">
      <ul class="pagination justify-content-center">
        <?php $prevDisabled = ($page <= 1) ? 'disabled' : ''; ?>
        <?php $nextDisabled = ($page >= $totalPages) ? 'disabled' : ''; ?>

        <li class="page-item <?=h($prevDisabled)?>">
          <a class="page-link" href="?<?=h(build_query_string(['page' => max(1, $page - 1)]))?>">Prev</a>
        </li>

        <?php
          $start = max(1, $page - 2);
          $end = min($totalPages, $page + 2);

          if ($start > 1) {
            echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>1])) . '">1</a></li>';
            if ($start > 2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
          }

          for ($p = $start; $p <= $end; $p++) {
            $active = ($p === $page) ? 'active' : '';
            echo '<li class="page-item ' . h($active) . '"><a class="page-link" href="?' . h(build_query_string(['page'=>$p])) . '">' . h($p) . '</a></li>';
          }

          if ($end < $totalPages) {
            if ($end < $totalPages - 1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
            echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>$totalPages])) . '">' . h($totalPages) . '</a></li>';
          }
        ?>

        <li class="page-item <?=h($nextDisabled)?>">
          <a class="page-link" href="?<?=h(build_query_string(['page' => min($totalPages, $page + 1)]))?>">Next</a>
        </li>
      </ul>

      <div class="text-center text-muted small">
        Total <?=h($totalRows)?> movements — Page <?=h($page)?> of <?=h($totalPages)?> — <?=h($perPage)?> rows/page
      </div>
    </nav>
  <?php endif; ?>

  <div class="text-muted small mt-2">
    Opening stock is estimated using current stock minus movements since the "From" date. If you have complete stock_moves history, it will be accurate.
  </div>
<?php endif; ?>

<?php include __DIR__ . '/../../partials/footer.php'; ?>
