<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_role(['ADMIN']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

$itemId = (int)($_GET['item_id'] ?? 0);
$dateFrom = trim($_GET['from'] ?? '');
$dateTo = trim($_GET['to'] ?? '');
if ($itemId <= 0 || $dateFrom === '' || $dateTo === '') {
  die('Missing parameters.');
}

$hasMoveCreatedAt = column_exists($pdo,'stock_moves','created_at');
$hasMoveType = column_exists($pdo,'stock_moves','move_type');

$hasPurchases = table_exists($pdo,'purchases');
$hasVendors = table_exists($pdo,'vendors') && $hasPurchases && column_exists($pdo,'purchases','vendor_id');
$hasRequests = table_exists($pdo,'requests') && table_exists($pdo,'users') && column_exists($pdo,'requests','requester_id');

// Item meta
$meta = $pdo->prepare("SELECT id,name,unit,stock_qty FROM items WHERE id=?");
$meta->execute([$itemId]);
$item = $meta->fetch();
if (!$item) die('Item not found');

// Opening
$sumSince = $pdo->prepare("
  SELECT COALESCE(SUM(qty),0) AS s
  FROM stock_moves
  WHERE item_id=? AND DATE(" . ($hasMoveCreatedAt ? "created_at" : "NOW()") . ") >= ?
");
$sumSince->execute([$itemId, $dateFrom]);
$movesSince = (float)($sumSince->fetchColumn() ?: 0);
$opening = (float)$item['stock_qty'] - $movesSince;

// Full movements
$mvSql = "
  SELECT
    " . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . " AS created_at,
    " . ($hasMoveType ? "sm.move_type" : "''") . " AS move_type,
    COALESCE(sm.ref_table,'') AS ref_table,
    COALESCE(sm.ref_id,0) AS ref_id,
    sm.qty,
    COALESCE(sm.note,'') AS note
    " . ($hasVendors ? ",
        CASE WHEN sm.ref_table='purchases' THEN COALESCE(v.name,'') ELSE '' END AS vendor_name
      " : ", '' AS vendor_name") . "
    " . ($hasRequests ? ",
        CASE WHEN sm.ref_table='requests' THEN COALESCE(req.full_name,'') ELSE '' END AS requester_name
      " : ", '' AS requester_name") . "
  FROM stock_moves sm
  " . ($hasPurchases ? "LEFT JOIN purchases p ON (sm.ref_table='purchases' AND p.id=sm.ref_id)" : "") . "
  " . ($hasVendors ? "LEFT JOIN vendors v ON v.id = p.vendor_id" : "") . "
  " . ($hasRequests ? "LEFT JOIN requests r ON (sm.ref_table='requests' AND r.id=sm.ref_id)
                       LEFT JOIN users req ON req.id = r.requester_id" : "") . "
  WHERE sm.item_id=?
    AND DATE(" . ($hasMoveCreatedAt ? "sm.created_at" : "NOW()") . ") BETWEEN ? AND ?
  ORDER BY " . ($hasMoveCreatedAt ? "sm.created_at" : "sm.ref_id") . " ASC, sm.id ASC
";
$mv = $pdo->prepare($mvSql);
$mv->execute([$itemId, $dateFrom, $dateTo]);
$rows = $mv->fetchAll();

// Build XLSX via simple HTML table (Excel-compatible) to avoid library dependency inside app
$filename = "stock_movement_item_{$itemId}_{$dateFrom}_{$dateTo}.xls";
header("Content-Type: application/vnd.ms-excel; charset=utf-8");
header("Content-Disposition: attachment; filename=\"$filename\"");

echo "<table border='1'>";
echo "<tr><th colspan='7'>Stock Movement Report</th></tr>";
echo "<tr><td><b>Item</b></td><td colspan='6'>" . htmlspecialchars($item['name']) . " (" . htmlspecialchars($item['unit']) . ")</td></tr>";
echo "<tr><td><b>From</b></td><td>" . htmlspecialchars($dateFrom) . "</td><td><b>To</b></td><td>" . htmlspecialchars($dateTo) . "</td><td><b>Opening</b></td><td colspan='2'>" . number_format($opening,0) . "</td></tr>";
echo "<tr><th>Date/Time</th><th>Type</th><th>Ref Table</th><th>Ref ID</th><th>Vendor</th><th>Requester</th><th>Qty</th><th>Note</th></tr>";

$net = 0.0;
foreach ($rows as $r) {
  $net += (float)$r['qty'];
  echo "<tr>";
  echo "<td>" . htmlspecialchars($r['created_at']) . "</td>";
  echo "<td>" . htmlspecialchars($r['move_type']) . "</td>";
  echo "<td>" . htmlspecialchars($r['ref_table']) . "</td>";
  echo "<td>" . htmlspecialchars((string)$r['ref_id']) . "</td>";
  echo "<td>" . htmlspecialchars($r['vendor_name'] ?? '') . "</td>";
  echo "<td>" . htmlspecialchars($r['requester_name'] ?? '') . "</td>";
  echo "<td>" . htmlspecialchars((string)$r['qty']) . "</td>";
  echo "<td>" . htmlspecialchars($r['note']) . "</td>";
  echo "</tr>";
}
$closing = $opening + $net;
echo "<tr><td colspan='6'><b>Net Movement</b></td><td colspan='2'>" . htmlspecialchars((string)$net) . "</td></tr>";
echo "<tr><td colspan='6'><b>Closing</b></td><td colspan='2'>" . htmlspecialchars((string)$closing) . "</td></tr>";
echo "</table>";
