\
<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['ADMIN']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}

if (!table_exists($pdo, 'vat_rates')) {
  include __DIR__ . '/../partials/header.php';
  echo '<div class="alert alert-warning shadow-sm">'
     . '<strong>VAT system not enabled yet.</strong><br>'
     . 'Run the SQL file in this zip: <code>sql/create_vat_rates.sql</code>, then refresh this page.'
     . '</div>';
  include __DIR__ . '/../partials/footer.php';
  exit;
}

// Actions: add, update, delete
$action = $_POST['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'add') {
  $vat = (float)($_POST['vat_percent'] ?? 0);
  $effective = trim($_POST['effective_from'] ?? '');
  $note = trim($_POST['note'] ?? '');

  if ($vat < 0 || $vat > 100) {
    flash_set('danger', 'VAT percent must be between 0 and 100.');
  } elseif ($effective === '') {
    flash_set('danger', 'Please select an effective date.');
  } else {
    try {
      $pdo->prepare("INSERT INTO vat_rates(vat_percent, effective_from, note, created_by) VALUES(?,?,?,?)")
          ->execute([$vat, $effective, ($note===''?null:$note), current_user()['id'] ?? null]);
      flash_set('success', 'VAT rate added successfully.');
    } catch (Throwable $e) {
      flash_set('danger', 'Could not add VAT rate: ' . $e->getMessage());
    }
  }
  header('Location: ' . base_url('admin/vat_rates.php'));
  exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'update') {
  $id = (int)($_POST['id'] ?? 0);
  $vat = (float)($_POST['vat_percent'] ?? 0);
  $effective = trim($_POST['effective_from'] ?? '');
  $note = trim($_POST['note'] ?? '');

  if ($id <= 0) {
    flash_set('danger', 'Invalid VAT row.');
  } elseif ($vat < 0 || $vat > 100) {
    flash_set('danger', 'VAT percent must be between 0 and 100.');
  } elseif ($effective === '') {
    flash_set('danger', 'Effective date is required.');
  } else {
    try {
      $pdo->prepare("UPDATE vat_rates SET vat_percent=?, effective_from=?, note=? WHERE id=?")
          ->execute([$vat, $effective, ($note===''?null:$note), $id]);
      flash_set('success', 'VAT rate updated.');
    } catch (Throwable $e) {
      flash_set('danger', 'Could not update VAT rate: ' . $e->getMessage());
    }
  }
  header('Location: ' . base_url('admin/vat_rates.php'));
  exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'delete') {
  $id = (int)($_POST['id'] ?? 0);
  if ($id <= 0) {
    flash_set('danger', 'Invalid VAT row.');
  } else {
    try {
      $pdo->prepare("DELETE FROM vat_rates WHERE id=?")->execute([$id]);
      flash_set('success', 'VAT rate deleted.');
    } catch (Throwable $e) {
      flash_set('danger', 'Could not delete VAT rate: ' . $e->getMessage());
    }
  }
  header('Location: ' . base_url('admin/vat_rates.php'));
  exit;
}

// Data
$currentVat = (float)($pdo->query("SELECT vat_percent FROM vat_rates WHERE effective_from <= CURDATE() ORDER BY effective_from DESC LIMIT 1")->fetchColumn() ?: 0);
$rows = $pdo->query("SELECT id, vat_percent, effective_from, note, created_at FROM vat_rates ORDER BY effective_from DESC, id DESC")->fetchAll();

include __DIR__ . '/../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">VAT Rates</h3>
    <div class="text-muted">Current VAT: <strong><?=h(number_format($currentVat,2))?>%</strong></div>
  </div>
</div>

<div class="card shadow-sm mb-4">
  <div class="card-body">
    <h5 class="mb-3">Add New VAT Rate</h5>
    <form method="post" class="row g-2 align-items-end">
      <input type="hidden" name="action" value="add">
      <div class="col-md-3">
        <label class="form-label">VAT %</label>
        <input class="form-control" type="number" name="vat_percent" step="0.01" min="0" max="100" required placeholder="e.g. 11.00">
      </div>
      <div class="col-md-3">
        <label class="form-label">Effective from</label>
        <input class="form-control" type="date" name="effective_from" required>
      </div>
      <div class="col-md-4">
        <label class="form-label">Note (optional)</label>
        <input class="form-control" name="note" placeholder="e.g. Government change">
      </div>
      <div class="col-md-2">
        <button class="btn btn-success w-100">Add</button>
      </div>
      <div class="form-text mt-2">Tip: Add a new row whenever VAT changes. Purchases can store VAT used for history.</div>
    </form>
  </div>
</div>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="width:80px;">ID</th>
            <th style="width:140px;">VAT %</th>
            <th style="width:160px;">Effective</th>
            <th>Note</th>
            <th style="width:190px;">Created</th>
            <th style="width:240px;"></th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="6" class="p-4 text-center text-muted">No VAT rates yet.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <tr>
              <td class="fw-semibold">#<?=h($r['id'])?></td>
              <td>
                <form method="post" class="d-flex gap-2">
                  <input type="hidden" name="action" value="update">
                  <input type="hidden" name="id" value="<?=h((int)$r['id'])?>">
                  <input class="form-control form-control-sm" type="number" step="0.01" min="0" max="100"
                         name="vat_percent" value="<?=h($r['vat_percent'])?>" style="max-width:120px;">
              </td>
              <td>
                  <input class="form-control form-control-sm" type="date"
                         name="effective_from" value="<?=h($r['effective_from'])?>" style="max-width:160px;">
              </td>
              <td>
                  <input class="form-control form-control-sm" name="note" value="<?=h($r['note'] ?? '')?>">
              </td>
              <td class="text-muted"><?=h($r['created_at'])?></td>
              <td class="text-end">
                  <button class="btn btn-sm btn-primary">Save</button>
                </form>

                <form method="post" class="d-inline" onsubmit="return confirm('Delete this VAT rate?');">
                  <input type="hidden" name="action" value="delete">
                  <input type="hidden" name="id" value="<?=h((int)$r['id'])?>">
                  <button class="btn btn-sm btn-outline-danger">Delete</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../partials/footer.php'; ?>
