<?php
session_start();

function base_path(string $rel): string {
  return __DIR__ . '/../' . ltrim($rel, '/');
}

function app_cfg(): array {
  static $cfg = null;
  if ($cfg !== null) return $cfg;

  $path = base_path('config/app.php');
  if (file_exists($path)) {
    $tmp = require $path;
    if (is_array($tmp)) { $cfg = $tmp; return $cfg; }
  }

  // Safe defaults (prevents white/500 pages if config/app.php is missing)
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  // Infer base folder from script name (expects /stock)
  $script = $_SERVER['SCRIPT_NAME'] ?? '/stock/index.php';
  $baseDir = '/' . trim(explode('/', trim($script,'/'))[0] ?? 'stock', '/');
  $cfg = [
    'base_url' => $scheme . '://' . $host . $baseDir,
    'site_url' => $scheme . '://' . $host,
    'app_name' => 'Stock',
    'admin_email' => '',
  ];
  return $cfg;
}

function base_url(string $path = ''): string {
  $cfg = app_cfg();
  $base = rtrim($cfg['base_url'] ?? '', '/');
  $path = ltrim($path, '/');
  return $base . '/' . $path;
}

function site_url(string $path = ''): string {
  $cfg = app_cfg();
  $site = rtrim($cfg['site_url'] ?? 'http://localhost', '/');
  return $site . base_url($path);
}

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $cfg = require base_path('config/db.php');
  $dsn = "mysql:host={$cfg['host']};dbname={$cfg['dbname']};charset={$cfg['charset']}";
  $pdo = new PDO($dsn, $cfg['user'], $cfg['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

function current_user(): ?array {
  return $_SESSION['user'] ?? null;
}

function require_login(): void {
  if (!current_user()) {
    header('Location: ' . base_url('auth/login.php'));
    exit;
  }
}

function require_role(array $roles): void {
  require_login();
  $u = current_user();
  if (!$u || !in_array($u['role'], $roles, true)) {
    http_response_code(403);
    echo "403 Forbidden";
    exit;
  }
}

function h(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function flash_set(string $type, string $msg): void {
  $_SESSION['flash'] = ['type' => $type, 'msg' => $msg];
}

function flash_get(): ?array {
  $f = $_SESSION['flash'] ?? null;
  unset($_SESSION['flash']);
  return $f;
}


/**
 * Activity logging (safe - never breaks pages)
 * Requires table activity_log (see SQL).
 */
function log_activity(string $action, ?string $entity = null, ?int $entity_id = null, ?string $details = null): void {
  try {
    $pdo = db();
    $u = current_user();
    $uid = $u ? (int)$u['id'] : null;
    $ip = $_SERVER['REMOTE_ADDR'] ?? null;

    $st = $pdo->prepare("INSERT INTO activity_log(user_id, action, entity, entity_id, details, ip_address)
                         VALUES(?,?,?,?,?,?)");
    $st->execute([$uid, $action, $entity, $entity_id, $details, $ip]);
  } catch (Throwable $e) {
    error_log('[ACTIVITY_LOG] ' . $e->getMessage());
  }
}
