<?php
require_once __DIR__ . '/bootstrap.php';
require_once __DIR__ . '/mailer.php';

/**
 * Absolute URL helper for emails (GLOBAL base URL)
 * Configure in: config/app.php  ->  base_url
 */
function _abs_url(string $path): string {
  $cfgPath = base_path('config/app.php');
  if (file_exists($cfgPath)) {
    $cfg = require $cfgPath;
    $base = rtrim((string)($cfg['base_url'] ?? ''), '/');
    if ($base !== '') {
      return $base . '/' . ltrim($path, '/');
    }
  }
  // Fallbacks
  if (function_exists('site_url')) return site_url($path);
  return base_url($path);
}

/**
 * Clean HTML email wrapper with logo + app name.
 * Configure in: config/app.php  -> app_name, logo_url
 */
function _email_layout(string $title, string $subtitle, string $html): string {
  $brand = 'Consumables System';
  $logo  = '';
  $cfgPath = base_path('config/app.php');
  if (file_exists($cfgPath)) {
    $cfg = require $cfgPath;
    $brand = (string)($cfg['app_name'] ?? $brand);
    $logo  = (string)($cfg['logo_url'] ?? $logo);
  }

  $now = date('Y-m-d H:i');

  $logoHtml = '';
  if ($logo !== '') {
    $logoHtml = '<img src="'.h($logo).'" alt="'.h($brand).'" style="height:40px;max-width:180px;object-fit:contain;">';
  }

  return '
  <div style="font-family:Arial,Helvetica,sans-serif;background:#f6f7fb;padding:24px;">
    <div style="max-width:720px;margin:0 auto;background:#ffffff;border:1px solid #e8e8ef;border-radius:12px;overflow:hidden;">
      <div style="padding:16px 20px;background:#111827;color:#ffffff;display:flex;align-items:center;gap:14px;">
        '.$logoHtml.'
        <div style="flex:1;">
          <div style="font-size:16px;font-weight:800;line-height:1.1;">'.h($brand).'</div>
          <div style="opacity:.85;font-size:12px;margin-top:2px;">'.$now.'</div>
        </div>
      </div>

      <div style="padding:22px;">
        <div style="font-size:18px;font-weight:800;color:#111827;margin-bottom:6px;">'.$title.'</div>
        <div style="font-size:13px;color:#6b7280;margin-bottom:18px;">'.$subtitle.'</div>
        '.$html.'
      </div>

      <div style="padding:14px 22px;background:#f9fafb;border-top:1px solid #eef2f7;color:#6b7280;font-size:12px;">
        This is an automated email from '.h($brand).'. Please do not reply to this email.
      </div>
    </div>
  </div>';
}

/** Load request + lines for email */
function _load_request_for_email(PDO $pdo, int $request_id): array {
  $st = $pdo->prepare("
    SELECT r.id, r.status, r.note, r.created_at, r.delivered_at,
           req.full_name AS requester_name, req.email AS requester_email
    FROM requests r
    JOIN users req ON req.id = r.requester_id
    WHERE r.id = ?
  ");
  $st->execute([$request_id]);
  $r = $st->fetch();
  if (!$r) return [];

  $lines = $pdo->prepare("
    SELECT i.name AS item_name, i.unit,
           rl.qty_requested, rl.qty_delivered
    FROM request_lines rl
    JOIN items i ON i.id = rl.item_id
    WHERE rl.request_id = ?
    ORDER BY i.name
  ");
  $lines->execute([$request_id]);
  $r['lines'] = $lines->fetchAll();

  return $r;
}

function _lines_table(array $lines, bool $showDelivered = false): string {
  $rows = '';
  foreach ($lines as $ln) {
    $item = h($ln['item_name']);
    $unit = h($ln['unit']);
    $req  = (int)$ln['qty_requested'];
    $del  = (int)$ln['qty_delivered'];

    $rows .= '<tr>
      <td style="padding:10px 10px;border-bottom:1px solid #eef2f7;"><strong>'.$item.'</strong><div style="color:#6b7280;font-size:12px;">'.$unit.'</div></td>
      <td style="padding:10px 10px;border-bottom:1px solid #eef2f7;text-align:right;">'.$req.'</td>';

    if ($showDelivered) {
      $rows .= '<td style="padding:10px 10px;border-bottom:1px solid #eef2f7;text-align:right;">'.$del.'</td>';
    }

    $rows .= '</tr>';
  }

  $thDelivered = $showDelivered
    ? '<th style="padding:10px 10px;border-bottom:1px solid #e5e7eb;text-align:right;">Delivered</th>'
    : '';

  return '
  <table cellspacing="0" cellpadding="0" style="width:100%;border-collapse:collapse;border:1px solid #e5e7eb;border-radius:10px;overflow:hidden;">
    <thead style="background:#f3f4f6;">
      <tr>
        <th style="padding:10px 10px;border-bottom:1px solid #e5e7eb;text-align:left;">Item</th>
        <th style="padding:10px 10px;border-bottom:1px solid #e5e7eb;text-align:right;">Requested</th>
        '.$thDelivered.'
      </tr>
    </thead>
    <tbody>'.$rows.'</tbody>
  </table>';
}

function _cta_button(string $url, string $label): string {
  $u = h($url);
  $l = h($label);
  return '<a href="'.$u.'" style="display:inline-block;background:#2563eb;color:#fff;text-decoration:none;padding:10px 14px;border-radius:10px;font-weight:800;font-size:13px;">'.$l.'</a>';
}

/**
 * Storekeepers: new request email with details (requester, note, items).
 * Auto mode: notify ALL storekeepers.
 */
function notify_storekeepers_new_request(int $request_id): void {
  $pdo = db();
  $r = _load_request_for_email($pdo, $request_id);
  if (!$r) {
    error_log("[NOTIFY] Request not found #{$request_id}");
    return;
  }

  $st = $pdo->prepare("
    SELECT id, full_name, email
    FROM users
    WHERE UPPER(role)='STOREKEEPER' AND email IS NOT NULL AND email<>''
    ORDER BY full_name
  ");
  $st->execute();
  $storekeepers = $st->fetchAll();
  if (!$storekeepers) {
    error_log("[NOTIFY] No storekeepers with email");
    return;
  }

  $viewUrl = _abs_url('storekeeper/request_view.php?id='.$request_id);

  $html = '
    <div style="margin-bottom:14px;">
      A new request has been submitted and is waiting to be delivered.
    </div>

    <div style="background:#f9fafb;border:1px solid #eef2f7;border-radius:12px;padding:14px;margin-bottom:14px;">
      <div style="display:flex;gap:18px;flex-wrap:wrap;">
        <div><div style="color:#6b7280;font-size:12px;">Request #</div><div style="font-weight:900;color:#111827;">'.h($request_id).'</div></div>
        <div><div style="color:#6b7280;font-size:12px;">Requester</div><div style="font-weight:900;color:#111827;">'.h($r['requester_name']).'</div></div>
        <div><div style="color:#6b7280;font-size:12px;">Created</div><div style="font-weight:900;color:#111827;">'.h($r['created_at']).'</div></div>
        <div><div style="color:#6b7280;font-size:12px;">Status</div><div style="font-weight:900;color:#b91c1c;">SUBMITTED</div></div>
      </div>
    </div>';

  if (!empty($r['note'])) {
    $html .= '<div style="margin-bottom:14px;"><div style="color:#6b7280;font-size:12px;margin-bottom:6px;">Note</div>
              <div style="background:#fff7ed;border:1px solid #fed7aa;border-radius:12px;padding:12px;">'.nl2br(h($r['note'])).'</div></div>';
  }

  $html .= '<div style="margin-bottom:12px;font-weight:900;color:#111827;">Items</div>';
  $html .= _lines_table($r['lines'], false);
  $html .= '<div style="margin-top:16px;">'._cta_button($viewUrl, 'Open Request').'</div>';

  $body = _email_layout(
    'New Request #'.h($request_id),
    'Please prepare and deliver the requested items.',
    $html
  );

  foreach ($storekeepers as $sk) {
    $to = trim((string)$sk['email']);
    if ($to === '') continue;
    send_email($to, "New request #{$request_id} (Consumables)", $body);
  }
}

/**
 * Requester: delivered email with delivered quantities table + link.
 */
function notify_requester_delivered(int $request_id): void {
  $pdo = db();
  $r = _load_request_for_email($pdo, $request_id);
  if (!$r) {
    error_log("[NOTIFY] Request not found #{$request_id}");
    return;
  }

  $to = trim((string)($r['requester_email'] ?? ''));
  if ($to === '') {
    error_log("[NOTIFY] Requester email missing for request #{$request_id}");
    return;
  }

  $viewUrl = _abs_url('requester/request_view.php?id='.$request_id);

  $html = '
    <div style="margin-bottom:14px;">
      Your request has been delivered. Please keep this email for reference.
    </div>

    <div style="background:#ecfdf5;border:1px solid #bbf7d0;border-radius:12px;padding:14px;margin-bottom:14px;">
      <div style="display:flex;gap:18px;flex-wrap:wrap;">
        <div><div style="color:#065f46;font-size:12px;">Request #</div><div style="font-weight:900;color:#065f46;">'.h($request_id).'</div></div>
        <div><div style="color:#065f46;font-size:12px;">Status</div><div style="font-weight:900;color:#065f46;">DELIVERED</div></div>
        <div><div style="color:#065f46;font-size:12px;">Delivered At</div><div style="font-weight:900;color:#065f46;">'.h($r['delivered_at'] ?? '').'</div></div>
      </div>
    </div>';

  $html .= '<div style="margin-bottom:12px;font-weight:900;color:#111827;">Delivered Items</div>';
  $html .= _lines_table($r['lines'], true);
  $html .= '<div style="margin-top:16px;">'._cta_button($viewUrl, 'View Request Details').'</div>';

  $body = _email_layout(
    'Request #'.h($request_id).' Delivered',
    'Items have been delivered by the storekeeper.',
    $html
  );

  send_email($to, "Your request #{$request_id} has been delivered", $body);
}
