<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['REQUESTER']);
$u = current_user();

$pdo = db();

// CHANGE THIS if your request form file has a different name:
$newRequestUrl = base_url('requester/new_request.php'); // common name
// Examples if your project uses different file names:
// $newRequestUrl = base_url('requester/request_new.php');
// $newRequestUrl = base_url('requester/create_request.php');
// $newRequestUrl = base_url('requester/request_form.php');

// ---------- Inputs ----------
$q = trim($_GET['q'] ?? '');
$statusFilter = trim($_GET['status'] ?? '');
$page = max(1, (int)($_GET['page'] ?? 1));

$perPageOptions = [5,10,20,50];
$perPage = (int)($_GET['per_page'] ?? 10);
if (!in_array($perPage, $perPageOptions, true)) $perPage = 10;

$allowedStatuses = ['SUBMITTED','PREPARING','DELIVERED','RECEIVED','CANCELLED'];
if (!in_array($statusFilter, $allowedStatuses, true)) $statusFilter = '';

// ---------- Helpers ----------
function status_badge_class(string $status): string {
  switch ($status) {
    case 'RECEIVED':  return 'bg-success';
    case 'PREPARING': return 'bg-warning text-dark';
    case 'SUBMITTED': return 'bg-danger';
    case 'DELIVERED': return 'bg-info text-dark';
    case 'CANCELLED': return 'bg-secondary';
    default:          return 'bg-dark';
  }
}

function build_query_string(array $params): string {
  $merged = array_merge($_GET, $params);
  foreach ($merged as $k => $v) {
    if ($v === '' || $v === null) unset($merged[$k]);
  }
  return http_build_query($merged);
}

function status_link(string $statusValue): string {
  $params = $_GET;
  $params['status'] = $statusValue;
  $params['page'] = 1;
  foreach ($params as $k => $v) {
    if ($v === '' || $v === null) unset($params[$k]);
  }
  return '?' . http_build_query($params);
}

// ---------- Summary Counters ----------
$summary = [
  'SUBMITTED' => 0, 'PREPARING' => 0, 'DELIVERED' => 0, 'RECEIVED' => 0, 'CANCELLED' => 0,
  'ALL' => 0
];

$sumStmt = $pdo->prepare("
  SELECT status, COUNT(*) AS c
  FROM requests
  WHERE requester_id = ?
  GROUP BY status
");
$sumStmt->execute([$u['id']]);
foreach ($sumStmt->fetchAll() as $row) {
  $st = $row['status'];
  if (isset($summary[$st])) $summary[$st] = (int)$row['c'];
  $summary['ALL'] += (int)$row['c'];
}

// ---------- Filtering WHERE ----------
$params = [$u['id']];
$where = "r.requester_id = ?";

if ($statusFilter !== '') {
  $where .= " AND r.status = ?";
  $params[] = $statusFilter;
}

if ($q !== '') {
  $where .= " AND (
      CAST(r.id AS CHAR) LIKE ?
      OR r.status LIKE ?
      OR sk.full_name LIKE ?
    )";
  $like = "%{$q}%";
  $params[] = $like;
  $params[] = $like;
  $params[] = $like;
}

// ---------- Total rows ----------
$countStmt = $pdo->prepare("
  SELECT COUNT(*) AS total
  FROM requests r
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  WHERE $where
");
$countStmt->execute($params);
$total = (int)$countStmt->fetch()['total'];
$totalPages = max(1, (int)ceil($total / $perPage));
if ($page > $totalPages) $page = $totalPages;
$offset = ($page - 1) * $perPage;

// ---------- List query ----------
$listSql = "
  SELECT
    r.id AS request_id,
    MAX(r.status) AS status,
    MAX(r.created_at) AS created_at,
    MAX(sk.full_name) AS storekeeper_name,

    COUNT(rl.id) AS lines_count,
    COALESCE(SUM(rl.qty_requested), 0) AS total_qty,

    COALESCE(
      GROUP_CONCAT(CONCAT(i.name, ' x', rl.qty_requested) ORDER BY i.name SEPARATOR ', '),
      ''
    ) AS items_summary

  FROM requests r
  LEFT JOIN users sk ON sk.id = r.assigned_to_id
  LEFT JOIN request_lines rl ON rl.request_id = r.id
  LEFT JOIN items i ON i.id = rl.item_id

  WHERE $where
  GROUP BY r.id
  ORDER BY r.id DESC
  LIMIT $perPage OFFSET $offset
";
$listStmt = $pdo->prepare($listSql);
$listStmt->execute($params);
$rows = $listStmt->fetchAll();

include __DIR__ . '/../partials/header.php';
?>

<style>
a.summary-card { text-decoration:none; color:inherit; display:block; }
a.summary-card:hover .card { transform: translateY(-1px); box-shadow: 0 .25rem .75rem rgba(0,0,0,.12); }
.summary-active .card { outline: 2px solid rgba(13,110,253,.45); }
</style>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">My Requests</h3>
    <div class="text-muted">Requester: <?=h($u['full_name'])?></div>
  </div>

  <div class="d-flex gap-2 align-items-center">
    <a class="btn btn-success btn-sm" href="<?=h($newRequestUrl)?>">+ New Request</a>

    <form method="get" class="d-flex gap-2 align-items-center">
      <input class="form-control form-control-sm" style="width:260px;"
             name="q" value="<?=h($q)?>" placeholder="Search: ID, status, storekeeper">

      <select class="form-select form-select-sm" name="status" style="width:170px;">
        <option value="" <?=($statusFilter===''?'selected':'')?>>All statuses</option>
        <option value="SUBMITTED" <?=($statusFilter==='SUBMITTED'?'selected':'')?>>SUBMITTED</option>
        <option value="PREPARING" <?=($statusFilter==='PREPARING'?'selected':'')?>>PREPARING</option>
        <option value="DELIVERED" <?=($statusFilter==='DELIVERED'?'selected':'')?>>DELIVERED</option>
        <option value="RECEIVED" <?=($statusFilter==='RECEIVED'?'selected':'')?>>RECEIVED</option>
        <option value="CANCELLED" <?=($statusFilter==='CANCELLED'?'selected':'')?>>CANCELLED</option>
      </select>

      <select class="form-select form-select-sm" name="per_page" style="width:140px;" onchange="this.form.submit()">
        <?php foreach ([5,10,20,50] as $opt): ?>
          <option value="<?=h($opt)?>" <?=($perPage===$opt?'selected':'')?>><?=h($opt)?> / page</option>
        <?php endforeach; ?>
      </select>

      <input type="hidden" name="page" value="1">
      <button class="btn btn-sm btn-primary">Apply</button>

      <?php if ($q !== '' || $statusFilter !== ''): ?>
        <a class="btn btn-sm btn-outline-secondary" href="<?=h(base_url('requester/my_requests.php'))?>">Clear</a>
      <?php endif; ?>
    </form>
  </div>
</div>

<div class="row g-2 mb-3">
  <div class="col-md-2 <?=($statusFilter===''?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link(''))?>">
      <div class="card shadow-sm">
        <div class="card-body py-2">
          <div class="text-muted small">All</div>
          <div class="fs-5 fw-semibold"><?=h($summary['ALL'])?></div>
        </div>
      </div>
    </a>
  </div>
  <div class="col-md-2 <?=($statusFilter==='SUBMITTED'?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link('SUBMITTED'))?>">
      <div class="card shadow-sm border-danger">
        <div class="card-body py-2">
          <div class="text-muted small">Submitted</div>
          <div class="fs-5 fw-semibold text-danger"><?=h($summary['SUBMITTED'])?></div>
        </div>
      </div>
    </a>
  </div>
  <div class="col-md-2 <?=($statusFilter==='PREPARING'?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link('PREPARING'))?>">
      <div class="card shadow-sm border-warning">
        <div class="card-body py-2">
          <div class="text-muted small">Preparing</div>
          <div class="fs-5 fw-semibold text-warning"><?=h($summary['PREPARING'])?></div>
        </div>
      </div>
    </a>
  </div>
  <div class="col-md-2 <?=($statusFilter==='DELIVERED'?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link('DELIVERED'))?>">
      <div class="card shadow-sm border-info">
        <div class="card-body py-2">
          <div class="text-muted small">Delivered</div>
          <div class="fs-5 fw-semibold text-info"><?=h($summary['DELIVERED'])?></div>
        </div>
      </div>
    </a>
  </div>
  <div class="col-md-2 <?=($statusFilter==='RECEIVED'?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link('RECEIVED'))?>">
      <div class="card shadow-sm border-success">
        <div class="card-body py-2">
          <div class="text-muted small">Received</div>
          <div class="fs-5 fw-semibold text-success"><?=h($summary['RECEIVED'])?></div>
        </div>
      </div>
    </a>
  </div>
  <div class="col-md-2 <?=($statusFilter==='CANCELLED'?'summary-active':'')?>">
    <a class="summary-card" href="<?=h(status_link('CANCELLED'))?>">
      <div class="card shadow-sm">
        <div class="card-body py-2">
          <div class="text-muted small">Cancelled</div>
          <div class="fs-5 fw-semibold"><?=h($summary['CANCELLED'])?></div>
        </div>
      </div>
    </a>
  </div>
</div>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <th style="width:90px;">ID</th>
            <th style="width:140px;">Status</th>
            <th style="width:220px;">Storekeeper</th>
            <th>Items Summary</th>
            <th style="width:150px;">Totals</th>
            <th style="width:190px;">Created</th>
            <th style="width:110px;"></th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="7" class="p-4 text-center text-muted">No requests found.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <?php
              $itemsSummary = $r['items_summary'] ?? '';
              if (strlen($itemsSummary) > 140) $itemsSummary = substr($itemsSummary, 0, 140) . '...';
              $sk = $r['storekeeper_name'] ?? '';
            ?>
            <tr>
              <td class="fw-semibold">#<?=h($r['request_id'])?></td>
              <td><span class="badge <?=h(status_badge_class($r['status']))?> px-3 py-2"><?=h($r['status'])?></span></td>
              <td><?=h($sk !== '' ? $sk : '—')?></td>
              <td class="text-muted"><?=h($itemsSummary !== '' ? $itemsSummary : '—')?></td>
              <td class="text-muted">
                <div><strong><?=h((int)$r['lines_count'])?></strong> lines</div>
                <div><strong><?=h((int)$r['total_qty'])?></strong> qty</div>
              </td>
              <td class="text-muted"><?=h($r['created_at'])?></td>
              <td class="text-end">
                <a class="btn btn-sm btn-outline-primary" href="<?=h(base_url('requester/request_view.php?id=' . $r['request_id']))?>">Open</a>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php if ($totalPages > 1): ?>
  <nav class="mt-3">
    <ul class="pagination justify-content-center">
      <?php $prevDisabled = ($page <= 1) ? 'disabled' : ''; ?>
      <?php $nextDisabled = ($page >= $totalPages) ? 'disabled' : ''; ?>

      <li class="page-item <?=h($prevDisabled)?>">
        <a class="page-link" href="?<?=h(build_query_string(['page' => max(1, $page - 1)]))?>">Prev</a>
      </li>

      <?php
        $start = max(1, $page - 2);
        $end = min($totalPages, $page + 2);

        if ($start > 1) {
          echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>1])) . '">1</a></li>';
          if ($start > 2) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
        }

        for ($p = $start; $p <= $end; $p++) {
          $active = ($p === $page) ? 'active' : '';
          echo '<li class="page-item ' . h($active) . '"><a class="page-link" href="?' . h(build_query_string(['page'=>$p])) . '">' . h($p) . '</a></li>';
        }

        if ($end < $totalPages) {
          if ($end < $totalPages - 1) echo '<li class="page-item disabled"><span class="page-link">…</span></li>';
          echo '<li class="page-item"><a class="page-link" href="?' . h(build_query_string(['page'=>$totalPages])) . '">' . h($totalPages) . '</a></li>';
        }
      ?>

      <li class="page-item <?=h($nextDisabled)?>">
        <a class="page-link" href="?<?=h(build_query_string(['page' => min($totalPages, $page + 1)]))?>">Next</a>
      </li>
    </ul>

    <div class="text-center text-muted small">
      Showing page <?=h($page)?> of <?=h($totalPages)?> — Total <?=h($total)?> requests — <?=h($perPage)?> per page
    </div>
  </nav>
<?php endif; ?>

<?php include __DIR__ . '/../partials/footer.php'; ?>
