<?php
require_once __DIR__ . '/../../lib/bootstrap.php';
require_once __DIR__ . '/../../lib/mailer.php';
require_role(['STOREKEEPER']);
$u = current_user();

$id = (int)($_POST['id'] ?? 0);
if ($id <= 0) {
  flash_set('danger', 'Invalid request id.');
  header('Location: ' . base_url('storekeeper/inbox.php'));
  exit;
}

$pdo = db();
$pdo->beginTransaction();

try {
  // Lock request row
  $stmt = $pdo->prepare("
    SELECT r.*, req.email AS requester_email, req.full_name AS requester_name
    FROM requests r
    JOIN users req ON req.id = r.requester_id
    WHERE r.id=? AND r.assigned_to_id=?
    FOR UPDATE
  ");
  $stmt->execute([$id, $u['id']]);
  $r = $stmt->fetch();

  if (!$r) throw new Exception('Request not found or not assigned to you.');
  if ($r['status'] !== 'PREPARING') throw new Exception('Request must be PREPARING.');

  // Lock lines + item stock
  $q = $pdo->prepare("
    SELECT rl.id AS line_id, rl.item_id, rl.qty_requested, rl.qty_delivered, i.stock_qty
    FROM request_lines rl
    JOIN items i ON i.id = rl.item_id
    WHERE rl.request_id=?
    FOR UPDATE
  ");
  $q->execute([$id]);
  $lines = $q->fetchAll();

  if (!$lines) throw new Exception('No request lines found.');

  // Default delivered=requested if not set; clamp delivered <= requested
  $setDelivered = $pdo->prepare("
    UPDATE request_lines
    SET qty_delivered=?
    WHERE id=? AND request_id=?
  ");

  foreach ($lines as &$ln) {
    $reqQty = (int)$ln['qty_requested'];
    $delQty = (int)$ln['qty_delivered'];

    if ($delQty <= 0) $delQty = $reqQty;     // auto-fill
    if ($delQty > $reqQty) $delQty = $reqQty; // prevent > requested

    $setDelivered->execute([$delQty, (int)$ln['line_id'], $id]);
    $ln['qty_delivered'] = $delQty; // keep updated for checks & deduction
  }
  unset($ln);

  // Validate stock
  foreach ($lines as $ln) {
    $deliver = max(0, (int)$ln['qty_delivered']);
    $stock = (int)$ln['stock_qty'];

    if ($deliver > $stock) {
      throw new Exception("Not enough stock for item_id {$ln['item_id']}. Reduce delivered qty or purchase stock.");
    }
  }

  // Deduct stock + insert stock moves
  $updItem = $pdo->prepare("UPDATE items SET stock_qty = stock_qty - ? WHERE id=?");
  $insMove = $pdo->prepare("
    INSERT INTO stock_moves(item_id, move_type, ref_table, ref_id, qty, created_by, note)
    VALUES(?, 'ISSUE_OUT', 'requests', ?, ?, ?, ?)
  ");

  foreach ($lines as $ln) {
    $deliver = max(0, (int)$ln['qty_delivered']);
    if ($deliver <= 0) continue;

    $updItem->execute([$deliver, (int)$ln['item_id']]);
    $insMove->execute([(int)$ln['item_id'], $id, -$deliver, (int)$u['id'], "Issued for request #{$id}"]);
  }

  // Update request status
  $pdo->prepare("
    UPDATE requests
    SET status='DELIVERED', delivered_at=NOW(), updated_at=NOW()
    WHERE id=?
  ")->execute([$id]);

  $pdo->commit();

  // Notify requester
  if (!empty($r['requester_email'])) {
    $body = "<p>Hello " . h($r['requester_name']) . ",</p>"
          . "<p>Your request #{$id} has been delivered. Please confirm received.</p>"
          . "<p>Open: <a href='" . h(site_url("requester/request_view.php?id={$id}")) . "'>Request #{$id}</a></p>";
    send_email($r['requester_email'], "Request #{$id} delivered", $body);
  }

  flash_set('success', 'Marked as DELIVERED. Stock deducted.');
  header('Location: ' . base_url("storekeeper/request_view.php?id={$id}"));
  exit;

} catch (Throwable $e) {
  $pdo->rollBack();
  flash_set('danger', 'Error: ' . $e->getMessage());
  header('Location: ' . base_url("storekeeper/request_view.php?id={$id}"));
  exit;
}
